<?php

namespace App\Console\Commands;

use App\Constants\Settings;
use Illuminate\Support\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Jobs\SendSmsJob;

class StudentFeeDueSms extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:student-fee-due-sms';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Command description';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Student fee due SMS job started');

        $month = Carbon::now()->month;
        $currentYear = Carbon::now()->year;

        $fromDate = Carbon::now()->startOfMonth()->format('Y-m-d');
        $toDate = Carbon::now()->endOfMonth()->format('Y-m-d');

        $this->info("Processing fees for month: $month, Year: $currentYear, From: $fromDate, To: $toDate");
        // ---------- Class-based Fees ----------
        $classFeesQuery = DB::table('class_fees')
            ->join('fee_types as ft', 'class_fees.fee_type_id', '=', 'ft.id')
            ->join('student_classes as sc', 'class_fees.student_class_id', '=', 'sc.id')
            ->join('student_infos as si', function ($join) {
                $join->on('sc.id', '=', 'si.student_class_id')
                    ->whereColumn('si.department', 'class_fees.department_id');
            })
            ->join('sections as sec', 'si.section_id', '=', 'sec.id')
            ->leftJoin('invoice_items as ii', function ($join) use ($fromDate, $toDate) {
                $join->on('si.id', '=', 'ii.student_id')
                    ->whereColumn('class_fees.fee_type_id', 'ii.fee_type_id')
                    ->whereBetween('ii.payment_date', [$fromDate, $toDate]);
            })
            ->where('si.status', 'active')
            ->whereNotIn('ft.code', ['Hostel', 'Transport'])
            ->when($month, fn ($q) => $q->whereIn('class_fees.month', [$month]))
            
            ->groupBy('si.id', 'si.name', 'si.student_id_no', 'sc.class_name', 'class_fees.department_id')
            ->select([
                'si.id',
                'si.name',
                'si.student_id_no',
                'si.branch_id',
                'si.mobile_no',
                'sc.class_name',
                DB::raw('SUM(class_fees.amount) as total_assigned_fee'),
                DB::raw('COALESCE(SUM(ii.amount), 0) as total_paid_fee'),
                DB::raw('SUM(class_fees.amount) - COALESCE(SUM(ii.amount), 0) as total_due_fee')
            ]);
        
        // $this->info("Class Fees Query: " . $classFeesQuery->toSql());
        // ---------- Student-specific Fees ----------
        $studentFeesQuery = DB::table('student_fees')
            ->join('fee_types as ft', 'student_fees.fee_type_id', '=', 'ft.id')
            ->join('student_infos as si', 'si.id', '=', 'student_fees.student_id')
            ->leftJoin('invoice_items as ii', function ($join) use ($fromDate, $toDate) {
                $join->on('student_fees.student_id', '=', 'ii.student_id')
                    ->whereColumn('student_fees.fee_type_id', 'ii.fee_type_id')
                    ->whereBetween('ii.payment_date', [$fromDate, $toDate]);
            })
            ->join('student_classes as sc', 'si.student_class_id', '=', 'sc.id')
            ->whereIn('ft.code', ['Hostel', 'Transport'])
            ->where('student_fees.status', 'active')
            ->where('si.status', 'active')
            ->groupBy('si.id', 'si.name', 'si.student_id_no', 'sc.class_name')
            ->select([
                'si.id',
                'si.name',
                'si.student_id_no',
                'si.branch_id',
                'si.mobile_no',
                'sc.class_name',
                DB::raw('SUM(student_fees.amount) as total_assigned_fee'),
                DB::raw('COALESCE(SUM(ii.amount), 0) as total_paid_fee'),
                DB::raw('SUM(student_fees.amount) - COALESCE(SUM(ii.amount), 0) as total_due_fee')
            ]);

        // ---------- Final Combined Query ----------
        $finalQuery = $classFeesQuery->unionAll($studentFeesQuery);

        $feeSummary = DB::query()
            ->fromSub($finalQuery, 'combined_fees')
            ->groupBy('id', 'name', 'student_id_no', 'class_name')
            ->select([
                'id',
                'name',
                'student_id_no',
                'class_name',
                'branch_id',
                'mobile_no',
                DB::raw('SUM(total_assigned_fee) as total_assigned_fee'),
                DB::raw('SUM(total_paid_fee) as total_paid_fee'),
                DB::raw('SUM(total_due_fee) as total_due_fee')
            ])
            ->havingRaw('SUM(total_due_fee) > 0') // Only show students who have due fees
            ->orderBy('name')
            ->get();
        $this->info("Fee Summary Count: " . $feeSummary->count());
        $successCount = 0;
        $failCount = 0;
        $invalidNumbers = [];
        $todayDay = now()->day;

        foreach ($feeSummary as $student) {
            // $this->info("Processing student: {$student->name} ({$student->student_id_no}) with due fee: {$student->total_due_fee}");
            $name = ltrim($student->name, ','); // remove leading comma if needed
            $studentId = $student->student_id_no;
            $class = $student->class_name;
            $number = $student->mobile_no;
            $dueAmount = $student->total_due_fee;

            $billing_due_sms = get_setting_value(Settings::BILLING_DUE_SMS,$student->branch_id) ? get_setting_value(Settings::BILLING_DUE_SMS,$student->branch_id) : '0';
            $billing_due_sms_date = get_setting_value(Settings::BILLING_DUE_SMS_DATE,$student->branch_id) ? get_setting_value(Settings::BILLING_DUE_SMS_DATE,$student->branch_id) : '0';
            $school_short_name = get_setting_value(Settings::SCHOOL_SHORT_NAME,$student->branch_id) ? get_setting_value(Settings::SCHOOL_SHORT_NAME,$student->branch_id) : 'N/A';

            // $this->info("billing_due_sms: $billing_due_sms, billing_due_sms_date: $billing_due_sms_date, school_short_name: $school_short_name");

            if ($billing_due_sms !== '1' || (int)$billing_due_sms_date !== $todayDay) {
                Log::info("Fee Due SMS skipped for student {$student->id} (branch {$student->branch_id})");
                continue;
            }

            if ($number) {
                // $this->info("student number: $number ");
                $smsBody = "Dear Guardian,\n";
                $smsBody .= "The nearest fee collection date of $name is 10-$month-$currentYear, due fee is Tk. $dueAmount. Please pay tuition fee Intime.\n";
                $smsBody .= "Regards,\n{$school_short_name}";
                // $this->info("smsBody : $smsBody ");
                if (function_exists('isValidBangladeshiNumber') && isValidBangladeshiNumber($number)) {
                    try {
                        SendSmsJob::dispatch($number, $smsBody, 'general', ['branch_id' => $student->branch_id]);
                        $successCount++;
                        $this->info("Birthday SMS dispatched to: {$student->name} ($number)");
                    } catch (\Exception $e) {
                        Log::error("SMS failed for {$number}: " . $e->getMessage());
                        $failCount++;
                    }
                } else {
                    $invalidNumbers[] = $number;
                    $failCount++;
                    $this->warn(" Invalid mobile number: $number");
                }
                Log::info("Success: $successCount, Fail: $failCount, Invalid Numbers: $invalidNumbers");
            }
        }
    }
}
