<?php

namespace App\Http\Controllers\Academic;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\Academic\Group;
use App\Models\Academic\Shift;
use App\Models\Academic\Section;
use App\Http\Controllers\Controller;
use App\Models\Academic\ClassSection;
use App\Models\Academic\StudentClass;
use Illuminate\Support\Facades\Validator;

class AcademicSetupController extends Controller
{
    
    public function index()
    {
        $shifts = Shift::all();
        $sections = Section::all();
        $groups = Group::all();
        $classSections = Section::get();
        $classShifts = Shift::pluck('shift_name', 'id');
        $studentClasses = StudentClass::with(['classSections.section', 'classSections.shift', 'classSections.classTeacher'])->get();
        return view('pages.academic.academic-setup.index',compact('classSections','classShifts','studentClasses', 'shifts','sections','groups'));
    }

    public function shift_store(Request $request)
    {
        $request->validate([
            'shift_name' => 'required|string|max:255|unique:shifts,shift_name',
            'start_time' => 'required',
            'end_time' => 'required',
        ]);
        Shift::create([
            'shift_name' => $request->shift_name,
            'start_time' => $request->start_time,
            'end_time' => $request->end_time,
        ]);
        return redirect()->back()->with('message','Shift Successfully Created');
    }

    public function shift_update(Request $request)
    {
        $request->validate([
            'shift_name' => 'required|string|max:255|unique:shifts,shift_name,' . $request->id,
            'start_time' => 'required',
            'end_time' => 'required',
        ]);
        Shift::where('id',$request->id)->update([
            'shift_name' => $request->shift_name,
            'start_time' => $request->start_time,
            'end_time' => $request->end_time,
        ]);
        return redirect()->back()->with('message','Shift Successfully Updated');
    }

    public function section_store(Request $request)
    {
        $request->validate([
            'section_name' => 'required|string|max:255|unique:sections,section_name',
        ]);

        Section::create([
            'section_name' => $request->section_name,
            'created_by' => auth()->id() ?? 1,
            'updated_by' => auth()->id() ?? 1,
        ]);
        return redirect()->back()->with('message','Section Successfully Created');
    }

    public function section_update(Request $request)
    {
        $request->validate([
            'section_name' => 'required|string|max:255|unique:sections,section_name,' . $request->id,
        ]);

        Section::where('id',$request->id)->update([
            'section_name' => $request->section_name,
            'created_by' => auth()->id() ?? NULL,
            'updated_by' => auth()->id() ?? NULL,
        ]);
        return redirect()->back()->with('message','Section Successfully Update');
    }


    public function group_store(Request $request)
    {

        $request->validate([
            'group_name' => 'required|string|max:255|unique:groups,group_name',
        ]);
        
        Group::create([
            'group_name' => $request->group_name,
            'status' => $request->status,
        ]);
        return redirect()->back()->with('success','Group Successfully Created');
    }


    public function group_update(Request $request)
    {
        $request->validate([
            'group_name' => 'required|string|max:255|unique:groups,group_name,' . $request->id,
        ]);
        
        Group::where('id',$request->id)->update([
            'group_name' => $request->group_name,
            'status' => $request->status,
        ]);
        return redirect()->back()->with('success','Group Successfully Updated');
    }

    

    public function student_class_store(Request $request)
    {
        $request->validate([
            'class_name' => 'required|string|max:255|unique:student_classes,class_name',
        ]);
    
        $sections = $request->sections ?? null;
        $shift_id = $request->shift_id ?? null;
        $branch_id = auth()->user()->branch_id;
    
        $classSlug = Str::slug($request->class_name);
    
        $existClass = StudentClass::where('slug', $classSlug)->first();
    
        if (!$existClass) {
            
            $existClass = StudentClass::create([
                'class_name' => $request->class_name,
                'slug' => $classSlug,
                'created_by' => auth()->id() ?? null,
            ]);
        }
    
        $class_sections = [];
        if ($sections) {
            foreach ($sections as $section) {

                if ($shift_id) {

                    $existingSection = ClassSection::where([
                        'student_class_id' => $existClass->id,
                        'section_id' => $section,
                        'shift_id' => $shift_id,
                        'branch_id' => $branch_id,
                    ])->first();
                } else {

                    $existingSection = ClassSection::where([
                        'student_class_id' => $existClass->id,
                        'section_id' => $section,
                        'branch_id' => $branch_id,
                    ])->first();
                }
    
                if ($existingSection) {
                    continue;
                }
    
                $class_sections[] = [
                    'branch_id' => $branch_id,
                    'shift_id' => $shift_id,
                    'student_class_id' => $existClass->id,
                    'section_id' => $section,
                ];
            }
        }
    
        // Insert the new sections if there are any
        if ($class_sections) {
            ClassSection::insert($class_sections);
        }
    
        return redirect()->back()->with('message', 'Class and sections successfully created or updated.');
    }  

    public function student_class_update(Request $request)
    {
        $request->validate([
            'class_name' => 'required|string|max:255|unique:student_classes,class_name,' . $request->id,
        ]);

        $sections = $request->sections ?? null;
        $shift_id = $request->shift_id ?? null;
        $branch_id = auth()->user()->branch_id;

        ClassSection::where('student_class_id', $request->id)->delete();

        StudentClass::where('id', $request->id)->update([
            'class_name' => $request->class_name,
            'updated_by' => auth()->id() ?? null,
        ]);
    
        $class_sections = [];
        if ($sections) {
            foreach ($sections as $section) {
                if ($shift_id) {
                    $existingSection = ClassSection::where([
                        'student_class_id' => $request->id,
                        'section_id' => $section,
                        'shift_id' => $shift_id,
                        'branch_id' => $branch_id,
                    ])->first();
                } else {
                    $existingSection = ClassSection::where([
                        'student_class_id' => $request->id,
                        'section_id' => $section,
                        'branch_id' => $branch_id,
                    ])->first();
                }
    
                if ($existingSection) {
                    continue;
                }
                $class_sections[] = [
                    'branch_id' => $branch_id,
                    'shift_id' => $shift_id,
                    'student_class_id' => $request->id,
                    'section_id' => $section,
                ];
            }
        }
    
        if ($class_sections) {
            ClassSection::insert($class_sections);
        }
    
        return redirect()->back()->with('message', 'Class successfully updated');
    }
    
}
