<?php

namespace App\Http\Controllers\Academic;

use App\Http\Controllers\Controller;
use App\Models\Academic\ClassRoutine;
use App\Models\Academic\RoutineDetail;
use App\Services\Academic\ClassRoutineService;
use Illuminate\Http\Request;

class ClassRoutineController extends Controller
{
    protected $classRoutineService;

    public function __construct(ClassRoutineService $classRoutineService)
    {
        $this->classRoutineService = $classRoutineService;
    }

    public function index()
    {
        $classRoutines = $this->classRoutineService->get();
        return view('pages.academic.class_routine.index', compact('classRoutines'));
    }

    public function create(Request $request)
    {

        $data = $this->classRoutineService->getRelatedData();
        $data['assignedSubjects'] = $this->classRoutineService->getAssignedSubjects($request->student_class);

        return view('pages.academic.class_routine.create')->with($data);
    }

    public function store(Request $request)
    {
        $request->validate([
            'days' => 'required|array',
            'subjects' => 'required|array',
            'subjects.*.*.subject' => 'required',
            'subjects.*.*.teacher' => 'required',
            'subjects.*.*.start_time' => 'required',
            'subjects.*.*.end_time' => 'required',
            'subjects.*.*.room_no' => 'required',
        ]);

        $this->classRoutineService->store($request);


        return redirect()->route('class-routine.index')->with('success', 'Class Routine updated successfully.');
    }

    public function show(ClassRoutine $classRoutine)
    {
        $classRoutine->load([
            'routineDetails' => function($query) {
                $query->orderBy('start_time', 'asc');
            },
            'routineDetails.teacher',
            'routineDetails.subject',
            'shift',
            'studentClass',
        ]);

        return view('pages.academic.class_routine.show', compact('classRoutine'));
    }

    public function edit($classRoutine)
    {
        $data = $this->classRoutineService->getRelatedData();
        $data['classRoutine'] = $this->classRoutineService->getSingleClassRoutine($classRoutine);
        $data['assignedSubjects'] = $this->classRoutineService->getAssignedSubjects($data['classRoutine']->student_class_id);
        return view('pages.academic.class_routine.edit', $data);
    }

    public function update(Request $request, ClassRoutine $classRoutine)
    {
        $request->validate([
            'days' => 'required|array',
            'subjects' => 'required|array',
            'subjects.*.*.subject' => 'required',
            'subjects.*.*.teacher' => 'required',
            'subjects.*.*.start_time' => 'required',
            'subjects.*.*.end_time' => 'required',
            'subjects.*.*.room_no' => 'required',
        ]);

        $this->classRoutineService->update($request, $classRoutine);

        return redirect()->route('class-routine.index')->with('success', 'Class time period updated successfully.');
    }

    public function destroy(ClassRoutine $classRoutine)
    {
        try {

            // Delete the class routine details
            RoutineDetail::where('class_routine_id', '=', $classRoutine->id)->delete();
            // Delete the class routine
            $classRoutine->delete();

            return redirect()->route('class-routine.index')->with('success', 'Class time period deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->route('class-routine.index')->with('error', 'Failed to delete class time period: ' . $e->getMessage());
        }
    }
}
