<?php

namespace App\Http\Controllers\Academic;

use Illuminate\Http\Request;
use App\Models\Academic\Subject;
use App\Models\HrPayroll\Employee;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\Academic\Section;
use App\Services\TeacherInfoService;
use App\Models\Academic\StudentClass;
use App\Models\Academic\SubjectsAssign;
use App\Models\Academic\TeacherEnrollment;

class TeacherEnrollmentController extends Controller
{
    protected $teacherInfoService;

    public function __construct(TeacherInfoService $teacherInfoService)
    {
        $this->teacherInfoService = $teacherInfoService;
    }

    public function index(){
        // $enrolls = DB::table('teacher_enrollments') // Replace with actual table name
        // ->select(
        //     'teacher_id',
        //     'student_classes.name as class_name',
        //     'subjects.subject_name as subject_name',
        //     'sections.name as section_name'
        // )
        // ->join('student_classes', 'teacher_enrollments.student_class_id', '=', 'student_classes.id')
        // ->join('subjects', 'teacher_enrollments.subject_id', '=', 'subjects.id')
        // ->join('sections', 'teacher_enrollments.section_id', '=', 'sections.id')
        // ->groupBy('teacher_id', 'student_classes.name', 'subjects.name', 'sections.name')
        // ->get();

        $enrolls = DB::table('teacher_enrollments')
            ->select(
                'teacher_enrollments.id',
                'teacher_enrollments.teacher_id',
                'employees.name as teacher_name',
                DB::raw("GROUP_CONCAT(DISTINCT CONCAT(subjects.subject_name, ' (', student_classes.class_name, ' ', sections.section_name, ')') ORDER BY student_classes.class_name, sections.section_name, subjects.subject_name SEPARATOR ', ') as subjects_detail")
            )
            ->join('student_classes', 'teacher_enrollments.student_class_id', '=', 'student_classes.id')
            ->join('subjects', 'teacher_enrollments.subject_id', '=', 'subjects.id')
            ->join('sections', 'teacher_enrollments.section_id', '=', 'sections.id')
            ->join('employees', 'teacher_enrollments.teacher_id', '=', 'employees.user_id')
            ->groupBy('teacher_enrollments.teacher_id')
            ->get();

        return view('pages.academic.teacher-enroll.index', compact('enrolls'));
    }

    public function create(){
        $classes = StudentClass::all();
        $subjects = Subject::all();
        $sections = Section::all();
        $mark_heads = DB::table('mark_heads')->get();
        $teachers = $this->teacherInfoService->getTeacherData();
        $subjectsAssigns = SubjectsAssign::with('studentClass', 'subject')->orderBy('student_class_id')->get()->groupBy('student_class_id');
        
        return view('pages.academic.teacher-enroll.create', compact('sections','subjectsAssigns','teachers'));
    }

    public function store(Request $request){
        try {
            $request->validate([
                'teacher_id' => 'required|exists:users,id',
            ]);
            $teacherId = $request->teacher_id;
            $branch_id = Employee::where('user_id',$teacherId)->first()->branch_id;

            if (!$request->has('class_subject_ids') || empty($request->class_subject_ids)) {
                return redirect()->back()->with('danger', 'Please select at least one subject.');
            }
        
            foreach ($request->class_subject_ids as $value) {
                // [$classId, $subjectId] = explode(',', $value);
                list($sectionId, $classId, $subjectId) = explode(',', $value);
                
                // Check if enrollment already exists
                $exists = TeacherEnrollment::where([
                    ['teacher_id', $teacherId],
                    ['student_class_id', $classId],
                    ['subject_id', $subjectId],
                    ['section_id', $sectionId],
                    ['branch_id', $branch_id],
                ])->exists(); // use exists() for better performance

                if ($exists) {
                    continue; // Skip to next loop iteration if already enrolled
                }
                
                $tEnroll = new TeacherEnrollment();
                $tEnroll->teacher_id = $teacherId;
                $tEnroll->student_class_id = $classId;
                $tEnroll->subject_id = $subjectId;
                $tEnroll->section_id = $sectionId;
                $tEnroll->branch_id = $branch_id;
                $tEnroll->save();
            }
        
            return redirect()->back()->with('success', 'Teacher Enrollment successfully completed.');
        } catch (\Exception $e) {
            return redirect()->back()->with('danger', 'Error: ' . $e->getMessage());
        }
    }

    public function edit($id){
        $classes = StudentClass::all();
        $subjects = Subject::all();
        $sections = Section::all();
        $mark_heads = DB::table('mark_heads')->get();
        $teachers = $this->teacherInfoService->getTeacherData();
        $subjectsAssigns = SubjectsAssign::with('studentClass', 'subject')->orderBy('student_class_id')->get()->groupBy('student_class_id');

        $enrollTeacherInfo = TeacherEnrollment::find($id);
        $enrollTeacherDatas = TeacherEnrollment::where('teacher_id',$enrollTeacherInfo->teacher_id)->get();
        $selectedEnrollments = $enrollTeacherDatas->map(function ($item) {
            return $item->section_id . ',' . $item->student_class_id . ',' . $item->subject_id;
        })->toArray();

        return view('pages.academic.teacher-enroll.edit', compact('sections','subjectsAssigns','teachers','enrollTeacherInfo','enrollTeacherDatas', 'selectedEnrollments'));
    }


    public function update(Request $request)
    {
        try {
            $request->validate([
                'teacher_id' => 'required|exists:users,id',
            ]);

            $teacherId = $request->teacher_id;
            TeacherEnrollment::where('teacher_id', $teacherId)->delete();
            $branch_id = Employee::where('user_id',$teacherId)->first()->branch_id;

            if (!$request->has('class_subject_ids') || empty($request->class_subject_ids)) {
                return redirect()->back()->with('danger', 'Please select at least one subject.');
            }
        
            foreach ($request->class_subject_ids as $value) {
                // [$classId, $subjectId] = explode(',', $value);
                list($sectionId, $classId, $subjectId) = explode(',', $value);
                
                // Check if enrollment already exists
                $exists = TeacherEnrollment::where([
                    ['teacher_id', $teacherId],
                    ['student_class_id', $classId],
                    ['subject_id', $subjectId],
                    ['section_id', $sectionId],
                    ['branch_id', $branch_id],
                ])->exists(); // use exists() for better performance

                if ($exists) {
                    continue; // Skip to next loop iteration if already enrolled
                }
                
                $tEnroll = new TeacherEnrollment();
                $tEnroll->teacher_id = $teacherId;
                $tEnroll->student_class_id = $classId;
                $tEnroll->subject_id = $subjectId;
                $tEnroll->section_id = $sectionId;
                $tEnroll->branch_id = $branch_id;
                $tEnroll->save();
            }
        
            return redirect()->route('teachers-enroll.index')
            ->with('success', 'Teacher enrollment updated successfully.');
        } catch (\Exception $e) {
            return redirect()->route('teachers-enroll.index')->with('danger', 'Error: ' . $e->getMessage());
        }
    }

    public function destroy($id){
        $enrollTeacherInfo = TeacherEnrollment::find($id);
        TeacherEnrollment::where('teacher_id', $enrollTeacherInfo->teacher_id)->delete();
        return redirect()->route('teachers-enroll.index')
            ->with('success', 'Teacher enrollment delete successfully.');
    }


}
