<?php

namespace App\Http\Controllers\Accounting;

use App\Services\Accounting\FeeSearchService;
use App\Http\Controllers\Controller;
use App\Http\Requests\FeeCollectionRequest;
use App\Models\Academic\AcademicYear;
use App\Models\Academic\StudentClass;
use App\Services\Accounting\FeeCollectionService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;

class FeeCollectController extends Controller
{
    protected FeeCollectionService $feeCollectionService;

    public function __construct(FeeCollectionService $feeCollectionService)
    {
        $this->feeCollectionService = $feeCollectionService;
    }

    public function index(Request $request)
    {
        // DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        // DB::table('student_fees')->truncate();
        // DB::table('invoice_items')->truncate();
        // DB::table('invoices')->truncate();
        // DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        $this->authorize('accounting.collect-fees.view');
        $data['collectedFees'] = $this->feeCollectionService->getInvoiceData($request->query());
        $data['studentClasses'] = StudentClass::pluck('class_name', 'id');
        $data['student_name'] = getStudentNameWithID($request->student_id_no);
        return view('pages.accounting.fee_collect.index', $data);
    }

    public function create(Request $request, FeeSearchService $feeSearchService)
    {
        $this->authorize('accounting.collect-fees.create');
        $data = $feeSearchService->getStudentFees($request);
        return view('pages.accounting.fee_collect.create', $data);
    }

    public function searchFees(Request $request, FeeSearchService $feeSearchService)
    {
        $data = $feeSearchService->getStudentFees($request);
        return view('pages.accounting.fee_collect.create', $data);
    }

    public function store(FeeCollectionRequest $request)
    {

        $this->authorize('accounting.collect-fees.create');

        try {
            $invoiceData = DB::transaction(function () use ($request) {
                return $this->feeCollectionService->collectFee($request);
            });
        } catch (Exception $e) {
            Log::error('Error collecting fee: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Error collecting fee!');
        }

        $action = $request->input('action');

        if ($action === 'collect-print') {

            $printData = $this->feeCollectionService->getSingleInvoiceData($invoiceData->uuid);
            return redirect()->route('collect_fee.show', $invoiceData->uuid)->with($printData);
        }

        return redirect()->route('collect_fee.index')->with('success', 'Fee Successfully Collected!');
    }

    public function show($uuid)
    {
        $this->authorize('accounting.collect-fees.view');
        $printData = $this->feeCollectionService->getSingleInvoiceData($uuid);
        return view('pages.accounting.fee_collect.show', $printData);
    }


    public function edit($uuid)
    {
        $this->authorize('accounting.collect-fees.view');
        $printData = $this->feeCollectionService->getSingleInvoiceData($uuid);

        return view('pages.accounting.fee_collect.edit', $printData);
    }


    public function destroy($invoice_id)
    {
        $this->authorize('accounting.collect-fees.delete');
        $this->feeCollectionService->delete($invoice_id);

        return redirect()->route('collect_fee.index')->with('success', 'Invoice deleted successfully.');
    }

    public function dueList(Request $request, FeeSearchService $feeSearchService)
    {
        $data = $feeSearchService->filterDueFeesSummary($request);
        // if ($request->ajax()) {
        //     return DataTables::of($data['dueFees'])
        //         ->addIndexColumn()
        //         ->make(true);
        // }

    
        $data['classes'] = StudentClass::pluck('class_name', 'id');
        $data['years'] = AcademicYear::orderBy('year', 'desc')->pluck('year', 'id');
        // dd($request->all(), $feeSearchService, $data);
        return view('pages.accounting.fee_collect.due', $data);
    }
}
