<?php

namespace App\Http\Controllers\Admission;

use Carbon\Carbon;
use App\Models\User;
use App\Jobs\SendSmsJob;
use App\Constants\Status;
use App\Constants\Settings;
use Illuminate\Support\Str;
use App\Models\GuardianInfo;
use Illuminate\Http\Request;
use App\Models\Academic\Shift;
use App\Models\Academic\Branch;
use App\Models\Student\Sibling;
use App\Models\Academic\Section;
use App\Models\Accounting\Invoice;
use App\Models\Student\StudentInfo;
use App\Services\FileUploadService;
use App\Http\Controllers\Controller;
use App\Models\HrPayroll\Occupation;
use App\Services\StudentInfoService;
use Illuminate\Support\Facades\Hash;
use App\Models\Academic\AcademicYear;
use App\Models\Academic\StudentClass;
use App\Models\Accounting\StudentFee;
use App\Models\Accounting\InvoiceItem;
use Yajra\DataTables\Facades\DataTables;
use App\Models\Admission\OnlineAdmission;
use App\Models\Guardian\GuardianRelation;
use App\Models\Admission\OnlineAdmissionSetting;
use Devfaysal\BangladeshGeocode\Models\Division;
use App\Services\Admission\OnlineAdmissionFeesService;

class OnlineAdmissionController extends Controller
{

    protected $studentInfoService;
    protected $fileUploadService;
    public function __construct(FileUploadService $fileUploadService, StudentInfoService $studentInfoService)
    {
        $this->fileUploadService = $fileUploadService;
        $this->studentInfoService = $studentInfoService;
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorize('online-admission.applicant-list');

        if ($request->ajax()) {
            $data = OnlineAdmission::with('studentClass:id,class_name')
                ->orderBy('id', 'desc');

            return DataTables::of($data)
                ->addIndexColumn()

                // 👇 Add combined full name
                ->addColumn('name', function ($row) {
                    return trim($row->first_name . ' ' . $row->last_name);
                })

                ->addColumn('class_name', function ($row) {
                    return $row->studentClass->class_name ?? '-';
                })
                ->addColumn('shift_name', function ($row) {
                    return $row->shift->shift_name ?? '-';
                })
                ->addColumn('action', function ($row) {
                    $viewUrl = route('online-admission.show', $row->id);
                    return '<div class="d-flex justify-content-center">
                                <a href="'.$viewUrl.'" class="btn btn-sm btn-soft-info" target="_blank" title="View">
                                    <i class="fas fa-eye"></i>
                                </a>
                            </div>';
                })

                ->rawColumns(['action'])

                // 👇 Custom sort & search
                ->filterColumn('form_id', function($query, $keyword) {
                    $query->where('form_id', 'like', "%{$keyword}%");
                })
                ->filterColumn('name', function($query, $keyword) {
                    $query->whereRaw("CONCAT(first_name, ' ', last_name) like ?", ["%{$keyword}%"]);
                })
                ->orderColumn('name', function ($query, $order) {
                    $query->orderByRaw("CONCAT(first_name, ' ', last_name) {$order}");
                })
                ->filterColumn('class_name', function($query, $keyword) {
                    $query->whereHas('studentClass', function($q) use ($keyword) {
                        $q->where('class_name', 'like', "%{$keyword}%");
                    });
                })
                ->filterColumn('shift_name', function($query, $keyword) {
                    $query->whereHas('shift', function($q) use ($keyword) {
                        $q->where('shift_name', 'like', "%{$keyword}%");
                    });
                })
                ->filterColumn('mobile_no', function($query, $keyword) {
                    $query->where('mobile_no', 'like', "%{$keyword}%");
                })

                ->make(true);
        }
        return view('pages.admission.online-admission.index');
    }

    public function applicantsToSelected(Request $request)
    {
        // $this->authorize('online-admission.applicant-to-selected');

        $ids = explode(',', $request->ids);
        // dd($ids, $request->all());
        OnlineAdmission::whereIn('id', $ids)
            ->update(['selected' => true]);

        return redirect()
            ->back()
            ->with('success', 'Selected applicants have been moved to the selected list successfully.');
    }

    public function selectedApplicantList(Request $request)
    {
        if ($request->ajax()) {
            $data = OnlineAdmission::with('studentClass:id,class_name')
                ->where('selected', 1)
                ->orderBy('id', 'desc');

            return DataTables::of($data)
                ->addIndexColumn()

                // 👇 Add combined full name
                ->addColumn('name', function ($row) {
                    return trim($row->first_name . ' ' . $row->last_name);
                })

                ->addColumn('class_name', function ($row) {
                    return $row->studentClass->class_name ?? '-';
                })
                ->addColumn('shift_name', function ($row) {
                    return $row->shift->shift_name ?? '-';
                })
                ->addColumn('action', function ($row) {
                    $viewUrl = route('online-admission.show', $row->id);
                    return '<div class="d-flex justify-content-center">
                                <a href="'.$viewUrl.'" class="btn btn-sm btn-soft-info" target="_blank" title="View">
                                    <i class="fas fa-eye"></i>
                                </a>
                            </div>';
                })

                ->rawColumns(['action'])

                // 👇 Custom sort & search
                ->filterColumn('form_id', function($query, $keyword) {
                    $query->where('form_id', 'like', "%{$keyword}%");
                })
                ->filterColumn('name', function($query, $keyword) {
                    $query->whereRaw("CONCAT(first_name, ' ', last_name) like ?", ["%{$keyword}%"]);
                })
                ->orderColumn('name', function ($query, $order) {
                    $query->orderByRaw("CONCAT(first_name, ' ', last_name) {$order}");
                })
                ->filterColumn('class_name', function($query, $keyword) {
                    $query->whereHas('studentClass', function($q) use ($keyword) {
                        $q->where('class_name', 'like', "%{$keyword}%");
                    });
                })
                ->filterColumn('shift_name', function($query, $keyword) {
                    $query->whereHas('shift', function($q) use ($keyword) {
                        $q->where('shift_name', 'like', "%{$keyword}%");
                    });
                })
                ->filterColumn('mobile_no', function($query, $keyword) {
                    $query->where('mobile_no', 'like', "%{$keyword}%");
                })

                ->make(true);
        }

        $shifts = Shift::pluck('shift_name', 'id');
        $sections = Section::pluck('section_name', 'id');
        // dd($shifts, $sections);
        return view('pages.admission.online-admission.selected-applicant-list', compact('shifts', 'sections'));
    }


    public function selectedToAdmitted(Request $request)
    {
        // dd($request->all(), 'selectedToAdmitted');   


        $generateStudentId = $this->studentInfoService->generateStudentId();
        
        $shift_id = $request->input('shift_id');
        $section_id = $request->input('section_id');
        
        $ids = explode(',', $request->input('ids'));
        // Example: Retrieve the applicants
        $applicants = OnlineAdmission::whereIn('id', $ids)->get();
      
        // dd($applicants, getDefaultAcademicYearValue());

        foreach($applicants as $applicant){
            // dd( $applicant );
            $branch_id = $applicant->branch_id ?? auth()->user()->branch_id;
            $first_name = $applicant->first_name;
            $last_name = $applicant->last_name ?? '';
            $applicant_mobile = $applicant->mobile_no;
            $blood_group = $applicant->blood_group;
            $nid = $applicant->nid ?? $applicant->birth_certi_no;
            $dob = $applicant->date_of_birth;
            $email = $applicant->email;
            $gender = $applicant->gender;
            $religion = $applicant->religion;
            $profile_image = $applicant->profile_image;
            $student_class_id = $applicant->student_class_id ?? null;

            $father_name = $applicant->father_name;
            $father_mobile = $applicant->father_mobile;
            $father_occupation = $applicant->father_occupation;

            $mother_name = $applicant->mother_name;
            $mother_mobile = $applicant->mother_mobile;
            $mother_occupation = $applicant->mother_occupation;



            $present_division_id = $applicant->present_division_id;
            $present_district_id = $applicant->present_district_id;
            $present_upazila_id = $applicant->present_upazila_id;
            $present_union_id = $applicant->present_union_id;
            $present_address = $applicant->present_address;

            $permanent_division_id = $applicant->permanent_division_id;
            $permanent_district_id = $applicant->permanent_district_id;
            $permanent_upazila_id = $applicant->permanent_upazila_id;
            $permanent_union_id = $applicant->permanent_union_id;
            $permanent_address = $applicant->permanent_address;

        
            $selected = $applicant->selected;
            $admitted = $applicant->admitted;

            // data store start
            if($selected == 1 && $admitted == 0){

                $lastRoll = StudentInfo::where('student_class_id', $student_class_id)->where('branch_id', auth()->user()->branch_id)->max('class_roll');

                $newRoll = $lastRoll ? $lastRoll + 1 : 1;
                
                if($applicant_mobile != null){
                    $checkUser = User::where('phone_number', $applicant_mobile)->first();
                        // dd($applicant_mobile, $checkUser);
                    if ($checkUser) {
                        dd($applicant_mobile, $checkUser);
                        return redirect()->back()->with('error', 'User with this mobile number already exists.');
                    }
                }
                // dd($applicant_mobile);
                
                $user_store = new User();
                $user_store->first_name = $first_name;
                $user_store->last_name = $last_name;
                $user_store->name = $first_name.' '.$last_name;
                $user_store->slug = Str::slug($first_name.'-'.$last_name);
                $user_store->user_type = 'student';
                $user_store->branch_id = $branch_id;
                $user_store->phone_number = $applicant_mobile;
                $user_store->email = $email;
                $user_store->password = Hash::make('12345678'); // Or generate random and send by email
                $user_store->save();
                
                $user_id = $user_store->id;
                
                $studentInfo = new StudentInfo();
                $studentInfo->student_id_no          = $generateStudentId ?? null; // or generate a unique one
                $studentInfo->user_id                = $user_id ?? null; // if available
                $studentInfo->branch_id              = $branch_id;
                $studentInfo->shift_id               = $shift_id;
                $studentInfo->section_id             = $section_id;
                $studentInfo->academic_year_id       = getDefaultAcademicYearID() ?? null;
                $studentInfo->first_name             = $first_name;
                $studentInfo->last_name              = $last_name;
                $studentInfo->name                   = $first_name . ' ' . $last_name;
                $studentInfo->age                    = null; // can be calculated from DOB if needed
                $studentInfo->gender                 = $gender;
                $studentInfo->blood_group            = $blood_group;
                $studentInfo->religion               = $religion;
                $studentInfo->date_of_birth          = $dob;
                $studentInfo->present_address        = $present_address;
                $studentInfo->permanent_address      = $permanent_address;
                $studentInfo->mobile_no              = $applicant_mobile;
                $studentInfo->email                  = $applicant->email ?? null;
                $studentInfo->birth_certi_no         = $nid;
                $studentInfo->profile_image          = $profile_image;
                $studentInfo->student_class_id       = $student_class_id;
                $studentInfo->version_id             = $applicant->version_id ?? 1;
                $studentInfo->session                = getDefaultAcademicYearValue();
                $studentInfo->admission_roll         = $applicant->form_id;
                $studentInfo->class_roll             = $newRoll;
                $studentInfo->year_admitted          = now()->format('Y'); // or custom
                $studentInfo->student_photo          = $profile_image;
                $studentInfo->dormitory_id           = $applicant->dormitory_id ?? null;
                $studentInfo->dorm_room_no           = $applicant->dorm_room_no ?? null;
                $studentInfo->father_name            = $father_name;
                $studentInfo->father_occupation      = $father_occupation;
                $studentInfo->father_mobile          = $father_mobile;
                $studentInfo->mother_name            = $mother_name;
                $studentInfo->mother_occupation      = $mother_occupation;
                $studentInfo->mother_mobile          = $mother_mobile;
                $studentInfo->guardian_name          = $applicant->guardian_name;
                $studentInfo->guardian_occupation    = $applicant->guardian_occupation ?? null;
                $studentInfo->guardian_mobile        = $applicant->guardian_mobile ?? null;
                $studentInfo->guardian_relationship  = $applicant->guardian_relation ?? null;
                $studentInfo->guardian_address       = $applicant->guardian_address ?? null;
                $studentInfo->present_division_id    = $present_division_id;
                $studentInfo->present_district_id    = $present_district_id;
                $studentInfo->present_upazila_id     = $present_upazila_id;
                $studentInfo->present_union_id       = $present_union_id;
                $studentInfo->permanent_division_id  = $permanent_division_id;
                $studentInfo->permanent_district_id  = $permanent_district_id;
                $studentInfo->permanent_upazila_id   = $permanent_upazila_id;
                $studentInfo->permanent_union_id     = $permanent_union_id;
                $studentInfo->created_by             = auth()->id();
                $studentInfo->updated_by             = auth()->id();
                $studentInfo->status                 = 'active'; // default value or use $applicant->status

                $studentInfo->save(); 

                $guardianInfo = new GuardianInfo();
                $guardianInfo->student_id             = $studentInfo->id;
                $guardianInfo->guardian_name          = $applicant->guardian_name ?? null;
                $guardianInfo->guardian_occupation    = $applicant->guardian_occupation ?? null;
                $guardianInfo->guardian_mobile        = $applicant->guardian_mobile ?? null;
                $guardianInfo->guardian_email         = $applicant->guardian_email ?? null;
                $guardianInfo->guardian_relationship  = $applicant->guardian_relation ?? null;
                $guardianInfo->guardian_address       = $applicant->guardian_address ?? null;
                $guardianInfo->password               = Hash::make('12345678');
                $guardianInfo->save();

                OnlineAdmission::where('id', $applicant->id)->update(['admitted' => 1]);
            }
        }

        return redirect()->back()->with('success', 'Student successfully admitted from selected applicants.');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {

        $studentClasses = StudentClass::whereIn('class_name', [
            'Play', 'Nursery', 'KG', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight'])->pluck('class_name', 'id');

        $shifts =  Shift::pluck('shift_name', 'id');
        $sections = Section::pluck('section_name', 'id');
        $branches = Branch::pluck('branch_name', 'id'); 
        $occupations = Occupation::pluck('name', 'id');
        $guardian_relations = GuardianRelation::pluck('name', 'id');
        $setting = OnlineAdmissionSetting::orderBy('id', 'desc')->first();
        $divisions = Division::all();
        $selectedDivision = 0;
        $selectedDistrict = 0;
        $selectedUpazila = 0;
        $selectedUnion = 0;

        return view('pages.admission.online-admission.create' ,  compact(
            'branches',
            'studentClasses',
            'shifts',
            'setting',
            'sections',
            'occupations',
            'guardian_relations',
            'divisions',
            'selectedDivision',
            'selectedDistrict',
            'selectedUpazila',
            'selectedUnion',
        ));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // $validatedData = $request->validate([
        //     'first_name' => 'required|string|max:255',
        //     'gender' => 'required|string|max:255',
        //     'religion' => 'required|string|max:255',
        //     'mobile_no' => 'required|string|max:255',
        //     // Add other validation rules as needed
        // ]);

        $data = $request->all();

        // ✅ Profile Image Upload
        if ($request->hasFile('profile_image')) {
            $file = $request->file('profile_image');
            $file_name = $this->fileUploadService->upload($file, 'uploads/student');
            $data['profile_image'] = $file_name;
        }

        // ✅ Transfer Certificate Upload
        if ($request->hasFile('previous_institute_tc')) {
            $file = $request->file('previous_institute_tc');
            $file_name = $this->fileUploadService->upload($file, 'uploads/online-admission/tc');
            $data['previous_institute_tc'] = $file_name;
        }

        // ✅ Create new admission with all data + file paths
        $onlineAdmission = OnlineAdmission::create($data);

        return redirect()
            ->back()
            ->with('success', 'Online admission form submitted successfully. Your Form ID is: ' . $onlineAdmission->form_id);
    }


    /**
     * Display the specified resource.
     */
    public function show( $id)
    {
        $studentClasses = StudentClass::whereIn('class_name', [
            'Play', 'Nursery', 'KG', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight'])->pluck('class_name', 'id');

        $shifts =  Shift::pluck('shift_name', 'id');
        $sections = Section::pluck('section_name', 'id');
        $branches = Branch::pluck('branch_name', 'id'); 
        $occupations = Occupation::pluck('name', 'id');
        $guardian_relations = GuardianRelation::pluck('name', 'id');
        $divisions = Division::all();
        $selectedDivision = 1;
        $selectedDistrict = 0;
        $selectedUpazila = 0;
        $selectedUnion = 0;


        $item = OnlineAdmission::findOrFail($id);
            // dd($item);
        return view('pages.admission.online-admission.show' ,  compact(
            'branches',
            'studentClasses',
            'shifts',
            'sections',
            'occupations',
            'guardian_relations',
            'divisions',
            'selectedDivision',
            'selectedDistrict',
            'selectedUpazila',
            'selectedUnion',
            'item'
        ));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(OnlineAdmission $onlineAdmission)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, OnlineAdmission $onlineAdmission)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(OnlineAdmission $onlineAdmission)
    {
        //
    }

    public function admittedList(Request $request)
    {
        $this->authorize('online-admission.qualified-list');
        if ($request->ajax()) {
            $data = OnlineAdmission::with('studentClass:id,class_name')
                ->where('admitted',1)
                ->orderBy('id', 'desc');

            return DataTables::of($data)
                ->addIndexColumn()

                // 👇 Add combined full name
                ->addColumn('name', function ($row) {
                    return trim($row->first_name . ' ' . $row->last_name);
                })

                ->addColumn('class_name', function ($row) {
                    return $row->studentClass->class_name ?? '-';
                })
                ->addColumn('shift_name', function ($row) {
                    return $row->shift->shift_name ?? '-';
                })
                ->addColumn('action', function ($row) {
                    $viewUrl = route('online-admission.show', $row->id);
                    return '<div class="d-flex justify-content-center">
                                <a href="'.$viewUrl.'" class="btn btn-sm btn-soft-info" target="_blank" title="View">
                                    <i class="fas fa-eye"></i>
                                </a>
                            </div>';
                })

                ->rawColumns(['action'])

                // 👇 Custom sort & search
                ->filterColumn('form_id', function($query, $keyword) {
                    $query->where('form_id', 'like', "%{$keyword}%");
                })
                ->filterColumn('name', function($query, $keyword) {
                    $query->whereRaw("CONCAT(first_name, ' ', last_name) like ?", ["%{$keyword}%"]);
                })
                ->orderColumn('name', function ($query, $order) {
                    $query->orderByRaw("CONCAT(first_name, ' ', last_name) {$order}");
                })
                ->filterColumn('class_name', function($query, $keyword) {
                    $query->whereHas('studentClass', function($q) use ($keyword) {
                        $q->where('class_name', 'like', "%{$keyword}%");
                    });
                })
                ->filterColumn('shift_name', function($query, $keyword) {
                    $query->whereHas('shift', function($q) use ($keyword) {
                        $q->where('shift_name', 'like', "%{$keyword}%");
                    });
                })
                ->filterColumn('mobile_no', function($query, $keyword) {
                    $query->where('mobile_no', 'like', "%{$keyword}%");
                })

                ->make(true);
        }

        // dd('ok');
        return view('pages.admission.online-admission.admitted-list');
    }

    public function selectedApplicantPayment(Request $request, $id, OnlineAdmissionFeesService $onlineAdmissionFeesService)
    {
        // $this->authorize('selected-applicant-payment');
        // dd($request);
        $data = $onlineAdmissionFeesService->getAdmissionAndFirstMonthFees($request);

        $applicant = OnlineAdmission::find($id);
        $data['applicant'] =  $applicant;

        // dd($data);
        return view('pages.admission.online-admission.selected-applicant-payment', $data);
    }

    public function selectedApplicantAdmitted(Request $request, $id)
    {
        $applicant  = OnlineAdmission::find($request->applicant_id);
        if(!$applicant){
            return redirect()->back()->with('error', 'Applicant not found.');
        }

        $yearLastTwoDigit = substr($applicant->session, 2, 2);
        
        $year_id = $request->year_id;
        $selected_month = $request->selected_month;

        $academicYear = AcademicYear::find($year_id);
        $year = $academicYear->year;

        $monthStartDate = date('Y-m-01', strtotime($selected_month . ' ' . $year));
        $monthEndDate = date('Y-m-t', strtotime($selected_month . ' ' . $year));

        $feeItems = json_decode($request->fee_types_and_amount, true);

        $password = rand(10000000, 99999999);

        $class_roll = admission_class_roll($applicant->student_class_id, $applicant->shift_id);
        // user create logic here
        $user = new User();
        $user->first_name = $applicant->first_name;
        $user->last_name = $applicant->last_name;
        $user->name = $applicant->first_name . ' ' . $applicant->last_name;
        $user->slug = Str::slug($applicant->first_name . '-' . $applicant->last_name);
        $user->user_type = 'student';
        $user->branch_id = $applicant->branch_id;
        $user->phone_number = $applicant->mobile_no;
        $user->email = $applicant->email;
        $user->password = Hash::make($password); 
        $user->profile_image_path = $applicant->profile_image; 
        $user->save();
        // student create logic here
        $student = new StudentInfo();
        $student->user_id = $user->id;
        $student->student_id_no = $this->studentInfoService->generateStudentId($yearLastTwoDigit);
        $student->academic_year_id = $year_id;

        $student->branch_id = $applicant->branch_id;
        $student->session = $applicant->session;

        $student->first_name = $applicant->first_name;
        $student->last_name = $applicant->last_name;
        $student->name = $applicant->first_name . ' ' . $applicant->last_name;

        $student->class_roll = $class_roll;
        $student->gender = $applicant->gender;
        $student->blood_group = $applicant->blood_group;
        $student->religion = $applicant->religion;
        $student->date_of_birth = $applicant->date_of_birth;
        $student->birth_certi_no = $applicant->birth_certi_no;
        $student->mobile_no = $applicant->mobile_no;
        $student->email = $applicant->email;
        $student->shift_id = $applicant->shift_id;
        $student->student_class_id = $applicant->student_class_id;
        $student->father_occupation = $applicant->father_occupation;
        $student->father_mobile = $applicant->father_mobile;
        $student->mother_name = $applicant->mother_name;
        $student->mother_occupation = $applicant->mother_occupation;
        $student->mother_mobile = $applicant->mother_mobile;
        $student->guardian_name = $applicant->guardian_name;
        $student->guardian_occupation = $applicant->guardian_occupation;
        $student->guardian_mobile = $applicant->guardian_mobile;
        $student->guardian_relationship = $applicant->guardian_relationship;
        $student->guardian_address = $applicant->guardian_address;
        $student->present_division_id = $applicant->present_division_id;
        $student->present_district_id = $applicant->present_district_id;
        $student->present_upazila_id = $applicant->present_upazila_id;
        $student->present_address = $applicant->present_address;
        $student->permanent_division_id = $applicant->permanent_division_id;
        $student->permanent_district_id = $applicant->permanent_district_id;
        $student->permanent_upazila_id = $applicant->permanent_upazila_id;
        $student->permanent_address = $applicant->permanent_address;
        $student->profile_image = $applicant->profile_image;
        $student->admission_roll = $applicant->form_id;
        $student->created_by = auth()->id();
        $student->save();

        // sibling logic here
        if(!empty($applicant->sibling_student_id_no)){
            $siblingStudent = StudentInfo::where('student_id_no', $applicant->sibling_student_id_no)->first();
            if(!empty($siblingStudent)){
                Sibling::insert([
                    'student_id' => $student->id,
                    'sibling_student_id_no' => $applicant->sibling_student_id_no,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }

        $GuardianInfo = new GuardianInfo();
        $GuardianInfo->student_id = $student->id;
        $GuardianInfo->guardian_name = $applicant->guardian_name;
        $GuardianInfo->guardian_occupation = $applicant->guardian_occupation;
        $GuardianInfo->guardian_mobile = $applicant->guardian_mobile;
        $GuardianInfo->guardian_relationship = $applicant->guardian_relationship;
        $GuardianInfo->guardian_address = $applicant->guardian_address;
        $GuardianInfo->password = Hash::make($password);
        $GuardianInfo->save();

        // sms sending logic here
        $student_create_sms = get_setting_value(Settings::STUDENT_CREATE_SMS) ? get_setting_value(Settings::STUDENT_CREATE_SMS) : '0';
        if($student_create_sms == '1'){
            $successCount = 0;
            $failCount = 0;
            $invalidNumbers = [];
            $number = $applicant->mobile_no;
            $school_short_name = get_setting_value(Settings::SCHOOL_SHORT_NAME) ? get_setting_value(Settings::SCHOOL_SHORT_NAME) : 'N/A';

            $studentClassName = $student->studentClass ? $student->studentClass->class_name : 'N/A';
            $studentSection = $student->section ? $student->section->section_name : 'N/A';
            $student_id = $student->student_id_no ?? 'N/A';

            $smsBody = "Dear ".$student->name.",\n";
            $smsBody .= "Your account has been created. You are now a student of Class ".$studentClassName.", Section ".$studentSection.".\n";
            $smsBody .= "Student ID: ".$student_id."\n";
            $smsBody .= "Password: ".$password."\n";
            $smsBody .= "Please keep this information safely.\n\n";
            $smsBody .= "Regards,\n";
            $smsBody .= $school_short_name;

            if (isValidBangladeshiNumber($number)) {
                SendSmsJob::dispatch($number, $smsBody, 'student', ['branch_id' => auth()->user()->branch_id]);
                $successCount++;
            } else {
                $invalidNumbers[] = $number;
                $failCount++;
            }
        }

        $applicant->update([
            'admitted'        => 1,
            'discount_type'   => $request->discount_type,
            'discount_amount' => $request->discount,
            'total_amount'    => $request->total_amount,
            'total_discount'  => $request->total_discount,
            'grand_total'     => $request->grand_total,
            'month'     => $selected_month,
            'invoice_date'     => Carbon::now()->format('Y-m-d'),
            'fee_details'     => json_encode($request->fee_details),
        ]);

        $invoiceData = [
            'student_info_id' => $student->id,
            'academic_year_id' => $year_id,
            'branch_id'   => $applicant->branch_id,
            'shift_id' => $student->shift_id,
            'student_class_id' => $student->student_class_id,
            'section_id' => $student->section_id,
            'total_amount' => $request->monthly_total_fees,
            'total_discount' =>  '0.00',
            'paid_amount' => $request->monthly_total_fees,
            'payment_date' => date('Y-m-d'),
            'invoice_date' => date('Y-m-d'),
            'due_date' => date('Y-m-d'),
            'payment_method' => $request->payment_method ?? NULL,
            'note' => $request->note ?? NULL,
            'created_by' => auth()->user()->id,
            'status' => Status::PAID
        ];

        $invoice_data = Invoice::create($invoiceData);

        $studentFee = StudentFee::create([
            'invoice_id' => $invoice_data->id,
            'branch_id' => $student->branch_id,
            'academic_year_id' => $year_id,
            'student_class_id' => $student->student_class_id,
            'student_id' => $student->id,
            'total_amount' => $request->monthly_total_fees,
            'total_discount' => '0.00',
            'grand_total' => $request->monthly_total_fees,
            'month' => $selected_month,
            'start_date' => $monthStartDate,
            'end_date' => $monthEndDate,
            'status' => Status::ACTIVE,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $invoiceItemData = [];
        foreach ($feeItems as $item) {
            [$fee_type_id, $amount] = explode('-', $item);
            $invoiceItemData[] = [
                'invoice_id' => $invoice_data->id,
                'student_id' => $student->id,
                'branch_id' => $student->branch_id,
                'fee_type_id' => (int) $fee_type_id,
                'amount' => (float) $amount,
                'month' => $selected_month,
                'academic_year_id' => $year_id,
                'student_fee_id' => $studentFee->id, //  Set StudentFee ID here
                'payment_date' => date('Y-m-d'),
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }
        InvoiceItem::insert($invoiceItemData);
        return redirect()->back()->with('success', 'Applicant has been admitted successfully.');
    }



    public function invoicePrint($id){
        $applicant = OnlineAdmission::find($id);
        return view('pages.admission.online-admission.invoice', compact('applicant'));
    }
}
