<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Services\Sms\SmsService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class ForgotPasswordController extends Controller
{
    private $smsService;

    public function __construct(SmsService $smsService)
    {
        $this->smsService = $smsService;
    }

    public function showForgotPasswordForm()
    {
        return view('auth.forgot-password');
    }

    public function requestOtp(Request $request)
    {
        $request->validate(['phone_number' => 'required|exists:users,phone_number']);

        $user = User::where('phone_number', $request->phone_number)->first();
        if (!$user) {
            return back()->withErrors(['phone_error' => 'User not found with this phone number.']);
        }

        $otp = random_int(100000, 999999);
        $user->otp = $otp;
        $user->otp_expires_at = Carbon::now()->addMinutes(2);
        $user->save();

        $userInfo = [
            'phone_number' => $request->phone_number,
            'user_id' => $user->id,
            'email' => $user->email,
            'branch_id' => $user->branch_id,
        ];
        // Simulate sending OTP via SMS
        $this->smsService->sendSms($request->phone_number, "Your EduEasy password reset OTP is, $otp. This OTP will expire in 2 minutes.", 'forgot-password', $userInfo);

        // Store phone number in session
        session(['phone_number' => $request->phone_number]);

        return redirect()->route('forgot-password.verify')->with('success', 'OTP sent successfully.');
    }

    public function showVerifyOtpForm(Request $request)
    {
        $phone_number = session('phone_number');
        if (!$phone_number) {
            return redirect()->route('forgot-password');
        }

        return view('auth.verify-otp', compact('phone_number'));
    }

    public function verifyOtp(Request $request)
    {
        $request->validate([
            'phone_number' => 'required|exists:users,phone_number',
            'otp' => 'required|digits:6',
        ]);

        $user = User::where('phone_number', $request->phone_number)->first();
        if (!$user) {
            return back()->withErrors(['phone_error' => 'User not found with this phone number.']);
        }

        // Check if OTP is expired
        if (Carbon::now()->greaterThan($user->otp_expires_at)) {
            return back()->withErrors(['otp' => 'OTP has expired. Please request a new one.']);
        }
        
        // Validate the OTP
        if ($user->otp != $request->otp) {
            return back()->withErrors(['otp' => 'Invalid OTP.']);
        }

        // Clear the OTP and expiration time after successful verification
        $user->otp = null; // Clear the OTP
        $user->otp_expires_at = null; // Clear the expiration time
        $user->save();

        // Redirect to the reset password form
        return redirect()->route('forgot-password.reset')->with('success', 'OTP verified successfully.');
    }

    public function showResetPasswordForm(Request $request)
    {
        $phone_number = session('phone_number');
        if (!$phone_number) {
            return redirect()->route('forgot-password');
        }

        return view('auth.reset-password', compact('phone_number'));
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'password' => 'required|confirmed|min:8',
        ]);

        $user = User::where('phone_number', session('phone_number'))->first();
        $user->password = Hash::make($request->password);
        $user->save();

        // Delete OTP record
        $user->otp = null;
        $user->otp_expires_at = null;
        $user->save();

        return redirect()->route('login')->with('success', 'Password reset successfully.');
    }

    public function updatePassword()
    {
        return view('auth.password.update');
    }

    public function updatePasswordSubmit(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'password' => 'required|confirmed|min:8',
            'password_confirmation' => 'required',
        ]);

        if (!Hash::check($request->input('current_password'), auth()->user()->password)) {
            return back()->withErrors(['current_password' => 'Current password does not match']);
        }
        $user = auth()->user();

        User::where('id', $user->id)->update([
            'password' => Hash::make($request->input('password'))
        ]);

        return redirect()->back()->with('status', 'Password updated successfully!');
    }
}
