<?php

namespace App\Http\Controllers\Certificate;

use App\Http\Controllers\Controller;
use App\Models\Certificate\CertificateTemplate;
use App\Services\FileUploadService;
use HTMLPurifier;
use HTMLPurifier_Config;
use Illuminate\Http\Request;

class CertificateTemplateController extends Controller
{
    protected $fileUploadService;

    public function __construct(FileUploadService $fileUploadService)
    {
        $this->fileUploadService = $fileUploadService;
    }

    public function index()
    {
        $templates = CertificateTemplate::all();

        return view('pages.certificate.templates.index', compact('templates'));
    }

    public function create()
    {
        return view('pages.certificate.templates.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required|string',
            'status' => 'in:active,inactive',
        ]);

        $upload_path = 'uploads/certificates';
        $imagePath = '';

        if ($request->hasFile('background_image')) {
            $file = $request->file('background_image');
            $attachment = $this->fileUploadService->upload($file, $upload_path);
            $imagePath = $attachment ?? null;
        }

        $config = HTMLPurifier_Config::createDefault();
        $purifier = new HTMLPurifier($config);
        $cleanHtml = $purifier->purify($request->body);
        $escapedHtml = htmlspecialchars($cleanHtml, ENT_QUOTES, 'UTF-8');

        CertificateTemplate::create([
            'title' => $request->title,
            'body' => $escapedHtml,
            'background_image' => $imagePath,
            'padding_x' => $request->padding_x,
            'padding_y' => $request->padding_y,
            'status' => $request->status,
        ]);

        return redirect()->route('certificate-templates.index')->with('success', 'Template created successfully.');
    }

    public function show(CertificateTemplate $certificateTemplate)
    {
        return view('pages.certificate.templates.show', compact('certificateTemplate'));
    }


    public function edit($id)
    {
        $certificateTemplate = CertificateTemplate::findOrFail($id);
        return view('pages.certificate.templates.edit', compact('certificateTemplate'));
    }

    public function update(Request $request, $id)
    {
        // Validate the request data
        $request->validate([
            'title' => 'required|string|max:255',
            'body' => 'required|string',
            'background_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'padding_x' => 'required|integer|min:0',
            'padding_y' => 'required|integer|min:0',
            'status' => 'required|in:active,inactive',
        ]);

        // Find the existing certificate template
        $certificateTemplate = CertificateTemplate::findOrFail($id);

        // Sanitize the HTML body
        $config = HTMLPurifier_Config::createDefault();
        $purifier = new HTMLPurifier($config);
        $cleanHtml = $purifier->purify($request->body);
        $escapedHtml = htmlspecialchars($cleanHtml, ENT_QUOTES, 'UTF-8');

        // Update the template data
        $certificateTemplate->title = $request->title;
        $certificateTemplate->body = $escapedHtml;
        $certificateTemplate->padding_x = $request->padding_x;
        $certificateTemplate->padding_y = $request->padding_y;
        $certificateTemplate->status = $request->status;

        // If a new background image is uploaded, store it
        $upload_path = 'uploads/certificates';

        if ($request->hasFile('background_image')) {
            $file = $request->file('background_image');
            $attachment = $this->fileUploadService->upload($file, $upload_path, $certificateTemplate->background_image);
            $certificateTemplate->background_image = $attachment;
        }


        // Save the updated template
        $certificateTemplate->save();

        // Redirect back with a success message
        return redirect()->route('certificate-templates.index')->with('success', 'Certificate template updated successfully.');
    }

    public function destroy(CertificateTemplate $certificateTemplate)
    {
        $certificateTemplate->delete();
        return redirect()->route('certificate_templates.index')->with('success', 'Template deleted successfully.');
    }
}
