<?php

namespace App\Http\Controllers;

use Carbon\CarbonPeriod;
use App\Models\Result\Exam;
use App\Models\Result\Grade;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use App\Services\Reports\ExamGradeCount;
use App\Services\Dashboard\DashboardService;

class DashboardController extends Controller
{

    protected $dashboardService;
    protected $examGradeCount;

    public function __construct(DashboardService $dashboardService, ExamGradeCount $examGradeCount)
    {
        $this->dashboardService = $dashboardService;
        $this->examGradeCount = $examGradeCount;
    }

    public function index()
    {

        $data = $this->dashboardService->getTotalStudents();
     
        if(auth()->user()->user_type == 'teacher') {

            $data['total_homework'] = $this->dashboardService->getTotalHomework();
            $data['submitted_homework'] = $this->dashboardService->getSubmittedHomework();
            
            // Ensure attendanceRecords is a collection
            $attendanceRecord = collect($this->dashboardService->getMyStudentAttendance());
            
            $attendanceRecords = $attendanceRecord['attendanceRecords'];
            $data['yesterday_present'] = $attendanceRecord['yesterday_present'];
            $data['total_present'] = $attendanceRecords->sum(function($student) {
                return $student?->attendance?->is_present ? 1 : 0;
            });
            $data['total_late'] = $attendanceRecords->sum(function($student) {
                return $student?->attendance?->is_late ? 1 : 0;
            });
            $data['total_absent'] = $attendanceRecords->sum(function($student) {
                return $student?->attendance?->is_absent ? 1 : 0;
            });
            
        
        } else {

            $data['total_teacher'] = $this->dashboardService->getTotalTeacher();
            $data['total_employee'] = $this->dashboardService->getTotalEmployee();
            $data['todays_collection'] = $this->dashboardService->getTodaysCollection();
            $data['this_month_collection'] = $this->dashboardService->getThisMonthCollection();
            $data['this_month_due'] = $this->dashboardService->getThisMonthDue();
            $data['this_month_income'] = $this->dashboardService->getThisMonthIncome();
            $data['this_month_expense'] = $this->dashboardService->getThisMonthExpense();
            $data['month_wise_income'] = $this->dashboardService->getMonthWiseIncome();
        }

        // Get chart data
        $data['classWiseSeries'] = $this->classWiseStudentPieChart();
        $data['genderSeries'] = $this->studentGenderPieChart();
        $data['examGradeSeries'] = $this->lastExamGradePieChart();
        $exam_id = getCurrentActiveExamId() ?? Exam::orderBy('id', 'desc')->first()->id; 
        $exam = Exam::find($exam_id); 
        $academic_year = $exam->academic_year->year ?? '';
        $data['examName'] = $exam->exam_name .' '.$academic_year ?? 'Last Exam';
        return view('layouts.dashboard', $data);
    }

    
    public function incomeExpenseCompare(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date_format:Y-m-d',
            'end_date' => 'required|date_format:Y-m-d',
        ]);

        $start = Carbon::createFromFormat('Y-m-d', $request->start_date);
        $end = Carbon::createFromFormat('Y-m-d', $request->end_date);

        // Generate date list
        $period = \Carbon\CarbonPeriod::create($start, $end);
        $dates = collect($period)->map(fn($date) => $date->toDateString());

        // Fetch and group income data
        $incomeData = DB::table('incomes')
            ->select(DB::raw('DATE(income_date) as date'), DB::raw('SUM(amount) as total'))
            ->whereBetween('income_date', [$start, $end])
            ->where('branch_id', auth()->user()->branch_id)
            ->groupBy('date')
            ->pluck('total', 'date'); // returns: ['2025-07-01' => 5000, ...]

        // Fetch and group expense data
        $expenseData = DB::table('expenses')
            ->select(DB::raw('DATE(expense_date) as date'), DB::raw('SUM(amount) as total'))
            ->whereBetween('expense_date', [$start, $end])
            ->where('branch_id', auth()->user()->branch_id)
            ->groupBy('date')
            ->pluck('total', 'date');

        // Map all dates with default 0 if no entry
        $income = $dates->map(fn($date) => (float) ($incomeData[$date] ?? 0));
        $expense = $dates->map(fn($date) => (float) ($expenseData[$date] ?? 0));

        return response()->json([
            'categories' => $dates,
            'series' => [
                ['name' => 'Income', 'data' => $income],
                ['name' => 'Expense', 'data' => $expense]
            ]
        ]);
    }



    // public function presentAbsentLateCompareOLD(Request $request)
    // {
    //     $request->validate([
    //         'start_date' => 'required|date_format:Y-m-d',
    //         'end_date' => 'required|date_format:Y-m-d',
    //     ]);
    //     $start = Carbon::createFromFormat('Y-m-d', $request->start_date);
    //     $end = Carbon::createFromFormat('Y-m-d', $request->end_date);

    //     // 🚫 Limit range to 60 days
    //     if ($start->diffInDays($end) > 60) {
    //         return response()->json([
    //             'error' => 'Date range too large. Please select within 60 days.'
    //         ], 422);
    //     }

    //     // 📅 Generate date list
    //     $period = CarbonPeriod::create($start, $end);
    //     $dates = collect($period)->map(fn($date) => $date->toDateString());

    //     // 🧠 Use cache to reduce DB load (optional)
    //     $cacheKey = 'attendance_' . auth()->id() . "_{$start->format('Ymd')}_{$end->format('Ymd')}";


    //     $attendanceData = Cache::remember($cacheKey, now()->addMinutes(10), function () use ($start, $end) {
    //         return DB::table('student_attendances')
    //             ->select(DB::raw('DATE(punch_date) as date'), 'day_status', DB::raw('COUNT(*) as total'))
    //             ->whereBetween('punch_date', [$start, $end])
    //             ->where('branch_id', auth()->user()->branch_id)
    //             ->whereIn('day_status', ['P', 'A']) // Present, Absent, Late
    //             ->groupBy(DB::raw('DATE(punch_date)'), 'day_status')
    //             ->get();
    //     });

    //     // 🔄 Normalize to [date][status] = total
    //     $normalized = [];
    //     foreach ($attendanceData as $row) {
    //         $normalized[$row->date][$row->day_status] = $row->total;
    //     }

    //     // 📊 Build series
    //     $present = [];
    //     $absent = [];

    //     foreach ($dates as $date) {
    //         $present[] = (float) ($normalized[$date]['P'] ?? 0);
    //         $absent[]  = (float) ($normalized[$date]['A'] ?? 0);
    //     }

    //     return response()->json([
    //         'categories' => $dates,
    //         'series' => [
    //             ['name' => 'Present', 'data' => $present],
    //             ['name' => 'Absent', 'data' => $absent],
    //         ]
    //     ]);
    // }

    public function presentAbsentLateCompare(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date_format:Y-m-d',
            'end_date'   => 'required|date_format:Y-m-d',
        ]);

        $start = Carbon::createFromFormat('Y-m-d', $request->start_date);
        $end   = Carbon::createFromFormat('Y-m-d', $request->end_date);

        // 🚫 Limit range to 60 days
        if ($start->diffInDays($end) > 60) {
            return response()->json([
                'error' => 'Date range too large. Please select within 60 days.'
            ], 422);
        }
        // 📅 Generate date list
        $period = CarbonPeriod::create($start, $end);
        $dates  = collect($period)->map(fn ($date) => $date->toDateString());

        // 📊 Fetch attendance data (NO CACHE)
        $attendanceData = DB::table('student_attendances')
            ->select(
                DB::raw('DATE(punch_date) as date'),
                'day_status',
                DB::raw('COUNT(*) as total')
            )
            ->whereBetween('punch_date', [$start, $end])
            ->where('branch_id', auth()->user()->branch_id)
            ->whereIn('day_status', ['P', 'A']) // Present, Absent
            ->groupBy(DB::raw('DATE(punch_date)'), 'day_status')
            ->get();

        // 🔄 Normalize data [date][status] = total
        $normalized = [];
        foreach ($attendanceData as $row) {
            $normalized[$row->date][$row->day_status] = $row->total;
        }


        // 📈 Build chart series
        $present = [];
        $absent  = [];

        foreach ($dates as $date) {
            $present[] = (float) ($normalized[$date]['P'] ?? 0);
            $absent[]  = (float) ($normalized[$date]['A'] ?? 0);
        }

        return response()->json([
            'categories' => $dates,
            'series' => [
                ['name' => 'Present', 'data' => $present],
                ['name' => 'Absent', 'data' => $absent],
            ]
        ]);
    }





    public function classWiseStudentPieChart()
    {
        $data = DB::table('student_infos')
        ->join('student_classes', 'student_infos.student_class_id', '=', 'student_classes.id')
        ->select('student_classes.class_name as name', DB::raw('count(*) as y'))
        ->where('student_infos.branch_id', auth()->user()->branch_id)
        ->groupBy('student_infos.student_class_id', 'student_classes.class_name')
        ->OrderBY('student_classes.id', 'asc')
        ->get();
        return $data->toArray();
    }

    public function studentGenderPieChart()
    {
        $data = DB::table('student_infos')
            ->select(
                'student_infos.gender as name',
                DB::raw('COUNT(*) as y')
            )
            ->where('student_infos.branch_id', auth()->user()->branch_id)
            ->groupBy('student_infos.gender')
            ->orderBy('y', 'desc')
            ->get()
            ->map(function ($item) {
                $item->name = ucfirst($item->name); // Capitalize first letter
                return $item;
            });
        return $data->toArray();
    }

    public function lastExamGradePieChart()
    {
        $studentGPAs = $this->examGradeCount->gradeCount(); // [student_id => GPA]
        // Load grades with GPA boundaries (descending to match highest first)
        $grades = Grade::select('name', 'gpa')->orderByDesc('gpa')->get();
        $gradeCounts = [];

        // Map GPA to grade name and count
        foreach ($studentGPAs as $gpa) {
            $grade = $grades->first(function ($g) use ($gpa) {
                return $gpa >= $g->gpa;
            });

            $gradeName = $grade ? $grade->name : 'N/A';

            if (!isset($gradeCounts[$gradeName])) {
                $gradeCounts[$gradeName] = 0;
            }

            $gradeCounts[$gradeName]++;
        }

        // Format data for Highcharts
        $data = collect($gradeCounts)->map(function ($count, $grade) {
            return ['name' => $grade, 'y' => $count];
        })->values();

        return $data->toArray();

    }

    public function examGradeChart(Request $request)
    {
        $examId = $request->exam_id;
        $studentGPAs = $this->examGradeCount->gradeCount($examId); // [student_id => GPA]
        $grades = Grade::select('name', 'gpa')->orderByDesc('gpa')->get();
        $gradeCounts = [];
        // Map GPA to grade name and count
        foreach ($studentGPAs as $gpa) {
            $grade = $grades->first(function ($g) use ($gpa) {
                return $gpa >= $g->gpa;
            });

            $gradeName = $grade ? $grade->name : 'N/A';

            if (!isset($gradeCounts[$gradeName])) {
                $gradeCounts[$gradeName] = 0;
            }

            $gradeCounts[$gradeName]++;
        }

        // Format data for Highcharts
        $data = collect($gradeCounts)->map(function ($count, $grade) {
            return ['name' => $grade, 'y' => $count];
        })->values();

        $exam = Exam::with('academic_year')->findOrFail($examId);

        return response()->json([
            'examName' => $exam->exam_name . ' ' . optional($exam->academic_year)->year,
            'series'   => $data->toArray(),
            'examId'   => $examId,
        ]);
    }

   
}
