<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Notice;
use App\Enums\NoticeEnum;
use Illuminate\Http\Request;
use App\Services\FileUploadService;
use App\Models\Academic\StudentClass;

class NoticeController extends Controller
{

    protected $fileUploadService;
    public function __construct(FileUploadService $fileUploadService){
        $this->fileUploadService = $fileUploadService;
    }

    public function index(){
        $notices = Notice::OrderBy('id','desc')->get();
        return view('pages.notice.index',compact('notices'));
    }

    public function create(){
        $studentClasses = StudentClass::pluck('class_name', 'id')->toArray(); 
        return view('pages.notice.create',compact('studentClasses'));
    }

    public function store(Request $request){
        $request->validate([
            'title' => 'required|string|max:250',
            'description' => 'nullable|string',
            'attachment' => 'nullable',
        ]);

        $notice = new Notice();
        $notice->title = $request->title;
        $notice->publish_date = Carbon::today()->format('Y-m-d');
        $notice->status = $request->status ?? NoticeEnum::Published;
        $notice->description = $request->description;
        $notice->created_by = auth()->id();
        $notice->branch_id = auth()->user()->branch_id;
        if($request->hasFile('attachment')){
            $file = $request->file('attachment');
            $file_name = $this->fileUploadService->upload($file,'uploads/notice');
            $notice->attachment = $file_name;
        }
        $notice->save();
        // Only attach if student_class_id is not empty
        if (!empty($request->student_class_id)) {
            $notice->studentClasses()->attach($request->student_class_id);
        }
        return redirect()->back()->with('success', 'Notice created successfully.');
    }

    public function show($id){
        $notice = Notice::findOrFail($id);
        return view('pages.notice.show',compact('notice'));
    }

    public function edit($id){
        $notice = Notice::findOrFail($id);
        $studentClasses = StudentClass::pluck('class_name','id');
        $selectedClassIds = $notice->studentClasses->pluck('id')->toArray();
        return view('pages.notice.edit',compact('notice','studentClasses','selectedClassIds'));
    }

    public function update(Request $request, $id){
        $notice = Notice::findOrFail($id);
        $request->validate([
            'title' => 'required|string|max:250',
            'description' => 'nullable|string',
        ]);
        $notice->title = $request->title;
        $notice->status = $request->status ?? NoticeEnum::Published;
        $notice->description = $request->description;
        $notice->updated_by = auth()->id();
        $notice->branch_id = auth()->user()->branch_id;
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $file_name = $this->fileUploadService->upload($file,'uploads/notice',$notice->attachment);
            $notice->attachment = $file_name;
        }
        $notice->save();
        $notice->studentClasses()->sync($request->student_class_id ?? []);
        return redirect()->route('notice.index')->with('success', 'Notice updated successfully.');
    }

    public function destroy($id)
    {
        $notice = Notice::findOrFail($id);
        $this->fileUploadService->delete($notice->attachment);
        $notice->studentClasses()->detach();
        $notice->delete();
        return redirect()->route('notice.index')->with('success', 'Notice deleted successfully.');
    }

}
