<?php

namespace App\Http\Controllers\Report;

use App\Http\Controllers\Controller;
use App\Models\Academic\AcademicYear;
use App\Models\Academic\Group;
use App\Models\Academic\Section;
use App\Models\Academic\Shift;
use App\Models\Academic\StudentClass;
use App\Models\Attendance\StudentAttendance;
use App\Models\Student\StudentInfo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Collection;
use App\Exports\StudentAttendanceExport;
use App\Exports\TypeWiseAttendance;
use Maatwebsite\Excel\Facades\Excel;
use App\Services\SettingService;
use App\Constants\Settings;
use App\Http\Requests\AttendanceReport;
use PDF;

use App\Services\Reports\StudentAttendanceReportService;
use Yajra\DataTables\Facades\DataTables;



class AttendanceReportController extends Controller
{
    protected $institute_name;
    protected $studentAttendanceReportService;

    public function __construct(StudentAttendanceReportService $studentAttendanceReportService)
    {
        $this->studentAttendanceReportService = $studentAttendanceReportService;
        $this->institute_name = get_setting_value(Settings::SCHOOL_NAME) ? get_setting_value(Settings::SCHOOL_NAME) : '';
    }

    public function index(){
        return view('report.attendance.index');
    }

    public function DateToDateReport(Request $request){
        $dates = null;
        $student_class_id = '';
        $shift_id = '';
        $studentClasses = StudentClass::pluck('class_name','id');
        $shifts = Shift::pluck('shift_name','id');
        return view('report.attendance.date-to-date-report',compact('dates','student_class_id','shift_id','studentClasses','shifts'));
    }

    public function dateToDateReportSearch(Request $request){
        try{
            // Parse the date range
            $daterange = $request->date;
            $from = substr($daterange, 0, -13);
            $from = date("Y-m-d", strtotime($from));
            $to = substr($daterange, -10);
            $to = date("Y-m-d", strtotime($to));
            // Fetch the selected columns or set defaults
            $selectedColumns = $request->input('selected_columns', []);
            if (empty($selectedColumns)) {
                $selectedColumns = ['shift_id','punch_date','day_status','is_present','is_absent','is_late'];
            }
            // Always include necessary columns
            $requiredColumns = ['id','student_id'];
            $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
            $columns = array_diff($selectColumns,['roll','student_name','total_present','total_absent','total_late','total_days']);

            // Build the query
            $studentAttendance = StudentAttendance::query();
            if($request->has('student_class_id')) {
                $studentAttendance->where('student_class_id', $request->input('student_class_id'));
            }
            if($request->has('shift_id')) {
                $studentAttendance->where('shift_id', $request->input('shift_id'));
            }
            if(!empty($from) && !empty($to)) {
                $studentAttendance->whereBetween('punch_date', [$from, $to]);
            }
            if(!empty($student_id)){
                $studentAttendance->with(['studentInfo' => function ($query) use ($student_id){
                    $query->where('student_id_no', $student_id);
                }]);
            }else{
                $studentAttendance->with(['studentInfo' ]);
            }
            $data = $studentAttendance
            ->select('student_id') // Select the unique column ->orderBy('class_roll','asc')
            ->selectRaw('SUM(student_attendances.is_present = 1) as total_present')
            ->selectRaw('SUM(student_attendances.is_absent = 1) as total_absent')
            ->selectRaw('SUM(student_attendances.is_late = 1) as total_late')
            ->groupBy('student_id'); // Group by the unique column
            // Return the DataTable
            return Datatables::of($data)
                ->addColumn('student_id', function ($data) {
                    return $data->StudentInfo->student_id_no ?? 'N/A';
                })
                ->addColumn('roll', function ($data) {
                    return $data->StudentInfo->class_roll ?? 'N/A';
                })
                ->addColumn('student_name', function ($data) {
                    return $data->StudentInfo->first_name.' '.$data->StudentInfo->last_name ?? 'N/A';
                })
                ->addColumn('total_present', function ($data) {
                    return $data->total_present ?? 'N/A';
                })
                ->addColumn('total_absent', function ($data) {
                    return $data->total_absent ?? 'N/A';
                })
                ->addColumn('total_late', function ($data) {
                    return $data->total_late ?? 'N/A';
                })
                ->addColumn('total_days', function ($data) {
                    $totalDays = ($data->total_present ?? 0) 
                                                    + ($data->total_absent ?? 0) 
                                                    + ($data->total_late ?? 0);

                    return $totalDays;
                })
                ->make(true);
        }catch (\Exception $e) {
            \Log::error('Error in date-to-date-report: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred. Please try again later.'], 500);
        }
    }
    public function dateToDateExportExcel(Request $request){ 
        $daterange = $request->date;
        $from = substr($daterange, 0, -13);
        $from = date("Y-m-d", strtotime($from));
        $to = substr($daterange, -10);
        $to = date("Y-m-d", strtotime($to));
        $class_name = StudentClass::find($request->student_class_id)->class_name ?? 'N/A';
        $shift_name = Shift::find($request->shift_id)->shift_name ?? 'N/A';

        // Fetch the selected columns or set defaults
        $selectedColumns = $request->input('selected_columns', []);
        if (empty($selectedColumns)) {
            $selectedColumns = ['shift_id','punch_date','day_status','is_present','is_absent','is_late'];
        }
        // Always include necessary columns
        $requiredColumns = ['id','student_id'];
        $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
        $columns = array_diff($selectColumns,['roll','student_name','total_present','total_absent','total_late','total_days']);

        $studentAttendance = StudentAttendance::query();
        if($request->has('student_class_id')) {
            $studentAttendance->where('student_class_id', $request->input('student_class_id'));
        }
        if($request->has('shift_id')) {
            $studentAttendance->where('shift_id', $request->input('shift_id'));
        }
        if(!empty($from) && !empty($to)) {
            $studentAttendance->whereBetween('punch_date', [$from, $to]);
        }
        if(!empty($student_id)){
            $studentAttendance->with(['studentInfo' => function ($query) use ($student_id){
                $query->where('student_id_no', $student_id);
            }]);
        }else{
            $studentAttendance->with(['studentInfo' ]);
        }
        $data = $studentAttendance
        ->select('student_id') // Select the unique column ->orderBy('class_roll','asc')
        ->selectRaw('SUM(student_attendances.is_present = 1) as total_present')
        ->selectRaw('SUM(student_attendances.is_absent = 1) as total_absent')
        ->selectRaw('SUM(student_attendances.is_late = 1) as total_late')
        ->groupBy('student_id')
        ->get();

        $data['class_name'] = $class_name;
        $data['shift_name'] = $shift_name;
        $data['form_date'] = $from;
        $data['to_date'] = $to;
        $data['institute_name'] = $this->institute_name;
        $data['report_title'] = 'Attendance Report';
        return Excel::download(new StudentAttendanceExport($data,$selectedColumns), 'date-to-date-attendance-report.xlsx');
    }
    public function dateToDateExportPDF(Request $request){
        $daterange = $request->date;
        $from = substr($daterange, 0, -13);
        $from = date("Y-m-d", strtotime($from));
        $to = substr($daterange, -10);
        $to = date("Y-m-d", strtotime($to));


        $class_name = StudentClass::find($request->student_class_id)->class_name ?? 'N/A';
        $shift_name = Shift::find($request->shift_id)->shift_name ?? 'N/A';
        // Fetch the selected columns or set defaults
        $selectedColumns = $request->input('selected_columns', []);
        if (empty($selectedColumns)) {
            $selectedColumns = ['shift_id','punch_date','day_status','is_present','is_absent','is_late'];
        }
        // Always include necessary columns
        $requiredColumns = ['id','student_id'];
        $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
        $columns = array_diff($selectColumns,['roll','student_name','total_present','total_absent','total_late','total_days']);

        $studentAttendance = StudentAttendance::query();
        if($request->has('student_class_id')) {
            $studentAttendance->where('student_class_id', $request->input('student_class_id'));
        }
        if($request->has('shift_id')) {
            $studentAttendance->where('shift_id', $request->input('shift_id'));
        }
        if(!empty($from) && !empty($to)) {
            $studentAttendance->whereBetween('punch_date', [$from, $to]);
        }
        if(!empty($student_id)){
            $studentAttendance->with(['studentInfo' => function ($query) use ($student_id){
                $query->where('student_id_no', $student_id);
            }]);
        }else{
            $studentAttendance->with(['studentInfo' ]);
        }
        $data = $studentAttendance
        ->select('student_id') // Select the unique column ->orderBy('class_roll','asc')
        ->selectRaw('SUM(student_attendances.is_present = 1) as total_present')
        ->selectRaw('SUM(student_attendances.is_absent = 1) as total_absent')
        ->selectRaw('SUM(student_attendances.is_late = 1) as total_late')
        ->groupBy('student_id')
        ->get();
       
        $pdf = Pdf::loadView('report.attendance.date-to-date-pdf', ['attendances'=>$data,'columns'=>$selectColumns,'class_name'=>$class_name,'shift_name'=>$shift_name,'form_date'=>$from,'to_date'=>$to,'report_title'=>'Attendance Report']);

        // $pdf = Pdf::loadView('report.attendance.date-to-date-pdf', ['attendances'=>$data,'columns'=>$selectColumns,'report_title'=>'Attendance Report']);
 
        return response()->streamDownload(function () use ($pdf) {
            echo $pdf->stream();
        }, 'users.pdf', [
            'Content-Type' => 'application/pdf',
        ]);
    }


    // attendance by type
    public function typeWiseAttendanceReport(Request $request){
        $dates = null;
        $attendance_type = '';
        $student_class_id = '';
        $shift_id = '';
        $studentClasses = StudentClass::pluck('class_name','id');
        $shifts = Shift::pluck('shift_name','id');
        return view('report.attendance.type-wise-attendance-report',compact('dates','attendance_type','student_class_id','shift_id','studentClasses','shifts'));
    }
    public function typeWiseAttendanceReportSearch(AttendanceReport $request){
        try {
            // Parse the date range
            $daterange = $request->date;
            $from = substr($daterange, 0, -13);
            $from = date("Y-m-d", strtotime($from));
            $to = substr($daterange, -10);
            $to = date("Y-m-d", strtotime($to));
            $attendance_type = $request->attendance_type;
            

            $type = null;
            $filter_col = null;
            if($request->attendance_type == "Present"){
                $type = "P";
                $filter_col = "is_present";
            }else if($request->attendance_type == "Absent"){
                $type = "A";
                $filter_col = "is_absent";
            }
            else if($request->attendance_type == "Late"){
                $type = "L";
                $filter_col = "is_late";
            }
            // Fetch the selected columns or set defaults
            $selectedColumns = $request->input('selected_columns', []);
            if (empty($selectedColumns)) {
                $selectedColumns = ['shift_id','punch_date','day_status','is_present','is_absent','is_late'];
            }
            // Always include necessary columns
            $requiredColumns = ['id','student_id'];
            $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
            $columns = array_diff($selectColumns,['roll','student_name','student_class','father_name','date_of_birth','gender','mobile_number','total_count']);

            // Build the query
            $studentAttendance = StudentAttendance::query();
            if($request->has('student_class_id')) {
                $studentAttendance->where('student_class_id', $request->input('student_class_id'));
            }
            if($request->has('shift_id')) {
                $studentAttendance->where('shift_id', $request->input('shift_id'));
            }
            if(!empty($from) && !empty($to)) {
                $studentAttendance->whereBetween('punch_date', [$from, $to]);
            }
            if(!empty($student_id)){
                $studentAttendance->with(['studentInfo' => function ($query) use ($student_id){
                    $query->where('student_id_no', $student_id);
                }]);
            }else{
                $studentAttendance->with(['studentInfo' ]);
            }
            $data = $studentAttendance
            ->select('student_id') // Select the unique column ->orderBy('class_roll','asc')
            ->selectRaw('SUM(student_attendances.'.$filter_col.' = 1) as total_count')
            ->groupBy('student_id'); // Group by the unique column
            // Return the DataTable
            return Datatables::of($data)
                ->addColumn('student_id', function ($data) {
                    return $data->StudentInfo->student_id_no ?? 'N/A';
                })
                ->addColumn('roll', function ($data) {
                    return $data->StudentInfo->class_roll ?? 'N/A';
                })
                ->addColumn('student_name', function ($data) {
                    return $data->StudentInfo->first_name.' '.$data->StudentInfo->last_name ?? 'N/A';
                })
                ->addColumn('student_class', function ($data) {
                    return $data->StudentInfo->studentClass->class_name ?? 'N/A';
                })
                ->addColumn('father_name', function ($data) {
                    return $data->StudentInfo->father_name ?? 'N/A';
                })
                ->addColumn('date_of_birth', function ($data) {
                    return $data->StudentInfo->date_of_birth ?? 'N/A';
                })
                ->addColumn('gender', function ($data) {
                    $gender = '<div class="text-capitalize">'.$data->StudentInfo->gender.'</div>';
                    return $gender ?? 'N/A';
                })
                ->addColumn('mobile_number', function ($data) {
                    return $data->StudentInfo->guardian_mobile ?? 'N/A';
                })
                ->addColumn('total_count', function ($data) {
                    return $data->total_count ?? 'N/A';
                })
                ->rawColumns(['gender']) 
                ->make(true);
        }
        catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        }
        catch (\Exception $e) {
            \Log::error('Error in type-wise-attendance-report: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred. Please try again later.'], 500);
        }
    }

    public function typeWiseAttendanceExportExcel(Request $request){ 
        $type = null;
        $filter_col = null;
        $class_name = null;
        $shift_name = null;
        $attendance_type = null;
        $daterange = $request->date;
        $from = substr($daterange, 0, -13);
        $from = date("Y-m-d", strtotime($from));
        $to = substr($daterange, -10);
        $to = date("Y-m-d", strtotime($to));
        $attendance_type = $request->attendance_type;
        $class_name = StudentClass::find($request->student_class_id)->class_name;
        $shift_name = Shift::find($request->shift_id)->shift_name;
        if($request->attendance_type == "Present"){
            $type = "P";
            $filter_col = "is_present";
        }else if($request->attendance_type == "Absent"){
            $type = "A";
            $filter_col = "is_absent";
        }
        else if($request->attendance_type == "Late"){
            $type = "L";
            $filter_col = "is_late";
        }
        // Fetch the selected columns or set defaults
        $selectedColumns = $request->input('selected_columns', []);
        if (empty($selectedColumns)) {
            $selectedColumns = ['shift_id','punch_date','day_status','is_present','is_absent','is_late'];
        }
        // Always include necessary columns
        $requiredColumns = ['id','student_id'];
        $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
        $columns = array_diff($selectColumns,['roll','student_name','student_class','father_name','date_of_birth','gender','mobile_number','total_count']);

        // Build the query
        $studentAttendance = StudentAttendance::query();
        if($request->has('student_class_id')) {
            $studentAttendance->where('student_class_id', $request->input('student_class_id'));
        }
        if($request->has('shift_id')) {
            $studentAttendance->where('shift_id', $request->input('shift_id'));
        }
        if(!empty($from) && !empty($to)) {
            $studentAttendance->whereBetween('punch_date', [$from, $to]);
        }
        if(!empty($student_id)){
            $studentAttendance->with(['studentInfo' => function ($query) use ($student_id){
                $query->where('student_id_no', $student_id);
            }]);
        }else{
            $studentAttendance->with(['studentInfo' ]);
        }
        $data = $studentAttendance
        ->select('student_id') // Select the unique column ->orderBy('class_roll','asc')
        ->selectRaw('SUM(student_attendances.'.$filter_col.' = 1) as total_count')
        ->groupBy('student_id') // Group by the unique column
        ->get();
        $data['class_name'] = $class_name;
        $data['shift_name'] = $shift_name;
        $data['attendance_type'] = $attendance_type;
        $data['form_date'] = $from;
        $data['to_date'] = $to;
        $data['institute_name'] = $this->institute_name;
        $data['report_title'] = 'Attendance Type Wise Report';

        return Excel::download(new TypeWiseAttendance($data,$selectedColumns), 'type-wise-attendance.xlsx');
    }

    public function typeWiseAttendanceExportPDF(AttendanceReport $request){
        $type = null;
        $filter_col = null;
        $class_name = null;
        $shift_name = null;
        $attendance_type = null;
        $daterange = $request->date;
        $from = substr($daterange, 0, -13);
        $from = date("Y-m-d", strtotime($from));
        $to = substr($daterange, -10);
        $to = date("Y-m-d", strtotime($to));
        $attendance_type = $request->attendance_type;
        $class_name = StudentClass::find($request->student_class_id)->class_name;
        $shift_name = Shift::find($request->shift_id)->shift_name;

        if($request->attendance_type == "Present"){
            $type = "P";
            $filter_col = "is_present";
        }else if($request->attendance_type == "Absent"){
            $type = "A";
            $filter_col = "is_absent";
        }
        else if($request->attendance_type == "Late"){
            $type = "L";
            $filter_col = "is_late";
        }
        // Fetch the selected columns or set defaults
        $selectedColumns = $request->input('selected_columns', []);
        if (empty($selectedColumns)) {
            $selectedColumns = ['shift_id','punch_date','day_status','is_present','is_absent','is_late'];
        }
        // Always include necessary columns
        $requiredColumns = ['id','student_id'];
        $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
        $columns = array_diff($selectColumns,['roll','student_name','student_class','father_name','date_of_birth','gender','mobile_number','total_count']);

        // Build the query
        $studentAttendance = StudentAttendance::query();
        if($request->has('student_class_id')) {
            $studentAttendance->where('student_class_id', $request->input('student_class_id'));
        }
        if($request->has('shift_id')) {
            $studentAttendance->where('shift_id', $request->input('shift_id'));
        }
        if(!empty($from) && !empty($to)) {
            $studentAttendance->whereBetween('punch_date', [$from, $to]);
        }
        if(!empty($student_id)){
            $studentAttendance->with(['studentInfo' => function ($query) use ($student_id){
                $query->where('student_id_no', $student_id);
            }]);
        }else{
            $studentAttendance->with(['studentInfo' ]);
        }
        $data = $studentAttendance
        ->select('student_id') // Select the unique column ->orderBy('class_roll','asc')
        ->selectRaw('SUM(student_attendances.'.$filter_col.' = 1) as total_count')
        ->groupBy('student_id') // Group by the unique column
        ->get();

        $pdf = Pdf::loadView('report.attendance.type-wise-pdf', ['attendances'=>$data,'columns'=>$selectColumns,'class_name'=>$class_name,'shift_name'=>$shift_name,'attendance_type'=>$attendance_type,'form_date'=>$from,'to_date'=>$to,'report_title'=>'Attendance Type Wise Report']);

        return response()->streamDownload(function () use ($pdf) {
            echo $pdf->stream();
        }, 'users.pdf', [
            'Content-Type' => 'application/pdf',
        ]);
    }

    // total attendances count by class and shift
    public function totalAttendanceReport(Request $request){
        $date = null;
        $attendance_type = '';
        $student_class_id = '';
        $shift_id = '';
        $studentClasses = StudentClass::pluck('class_name','id');
        $shifts = Shift::pluck('shift_name','id');
        return view('report.attendance.total-attendance-report',compact('date'));
    }
    public function totalAttendanceReportSearch(Request $request){
        try{
            // Parse the selected date
            $date = $request->date;
            if (empty($date)) {
                return response()->json(['error' => 'Date is required'], 400);
            }
            $date = date("Y-m-d", strtotime($date));
            // Fetch the selected columns or set defaults
            $selectedColumns = $request->input('selected_columns', []);
            if (empty($selectedColumns)) {
                $selectedColumns = ['class_name', 'shift_name', 'total_student', 'total_present', 'total_absent', 'total_late'];
            }
            // Always include necessary columns
            $requiredColumns = ['id', 'class_name', 'shift_name'];
            $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
            $columns = array_diff($selectColumns, ['total_student', 'total_present', 'total_absent', 'total_late']);
            // Build the query
            $classAttendance = StudentAttendance::query()
                ->join('student_classes', 'student_attendances.student_class_id', '=', 'student_classes.id') // Join with classes table
                ->join('shifts', 'student_attendances.shift_id', '=', 'shifts.id') // Join with shifts table
                ->select(
                    'student_classes.class_name', // Class name
                    'shifts.shift_name', // shift name
                    \DB::raw('COUNT(student_attendances.student_id) as total_student'), // Total students
                    \DB::raw('SUM(student_attendances.is_present = 1) as total_present'), // Total present
                    \DB::raw('SUM(student_attendances.is_absent = 1) as total_absent'), // Total absent
                    \DB::raw('SUM(student_attendances.is_late = 1) as total_late') // Total late
                )
                ->whereDate('student_attendances.punch_date', $date) // Filter by date
                ->groupBy('student_attendances.student_class_id', 'student_attendances.shift_id') // Group by class and shifts
                ->orderBy('student_classes.class_name', 'desc') // Order by class name
                ->orderBy('shifts.shift_name', 'asc') // Order by shifts name
                ->get();
            // Return the result as a DataTable
            return Datatables::of($classAttendance)
                ->addColumn('class_name', function ($data) {
                    return $data->class_name;
                })
                ->addColumn('shift_name', function ($data) {
                    return $data->shift_name;
                })
                ->addColumn('total_student', function ($data) {
                    return $data->total_student;
                })
                ->addColumn('total_present', function ($data) {
                    return $data->total_present;
                })
                ->addColumn('total_absent', function ($data) {
                    return $data->total_absent;
                })
                ->addColumn('total_late', function ($data) {
                    return $data->total_late;
                })
                ->make(true);
        }
        catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } 
        catch(\Exception $e){
            \Log::error('Error in type-wise-attendance-report: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred. Please try again later.'], 500);
        }
    }
    public function totalAttendanceExportPDF(Request $request){
        $date = $request->date;
        $filter_date = date("d-m-Y", strtotime($date));
        if (empty($date)) {
            return response()->json(['error' => 'Date is required'], 400);
        }
        $date = date("Y-m-d", strtotime($date));
        // Fetch the selected columns or set defaults
        $selectedColumns = $request->input('selected_columns', []);
        if (empty($selectedColumns)) {
            $selectedColumns = ['class_name', 'shift_name', 'total_student', 'total_present', 'total_absent', 'total_late'];
        }
        // Always include necessary columns
        $requiredColumns = ['id', 'class_name', 'shift_name'];
        $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
        $columns = array_diff($selectColumns, ['total_student', 'total_present', 'total_absent', 'total_late']);
        // Build the query
        $data = StudentAttendance::query()
                ->join('student_classes', 'student_attendances.student_class_id', '=', 'student_classes.id') // Join with classes table
                ->join('shifts', 'student_attendances.shift_id', '=', 'shifts.id') // Join with shifts table
                ->select(
                    'student_classes.class_name', // Class name
                    'shifts.shift_name', // shift name
                    \DB::raw('COUNT(student_attendances.student_id) as total_student'), // Total students
                    \DB::raw('SUM(student_attendances.is_present = 1) as total_present'), // Total present
                    \DB::raw('SUM(student_attendances.is_absent = 1) as total_absent'), // Total absent
                    \DB::raw('SUM(student_attendances.is_late = 1) as total_late') // Total late
                )
                ->whereDate('student_attendances.punch_date', $date) // Filter by date
                ->groupBy('student_attendances.student_class_id', 'student_attendances.shift_id') // Group by class and shifts
                ->orderBy('student_classes.class_name', 'desc') // Order by class name
                ->orderBy('shifts.shift_name', 'asc') // Order by shifts name
                ->get();
        $pdf = Pdf::loadView('report.attendance.total-attendance-pdf', ['classAttendance'=>$data,'filter_date'=>$filter_date,'columns'=>$selectedColumns,'report_title'=>'Total Attendance Report']);
        return response()->streamDownload(function () use ($pdf) {
            echo $pdf->stream();
        }, 'users.pdf', [
            'Content-Type' => 'application/pdf',
        ]);
    }
}