<?php

namespace App\Http\Controllers\Report;

use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Http\Controllers\Controller;
use App\Exports\ExportStudentHostel;
use Illuminate\Http\Request;
use App\Models\Hostel\RoomType;
use App\Models\Hostel\Hostel;
use App\Models\Hostel\HostelFloor;
use App\Models\Hostel\StudentHostel;
use App\Models\Student\StudentInfo;
use App\Services\SettingService;
use App\Constants\Settings;
use PDF;

class HostelReportController extends Controller
{

    protected $institute_name;

    public function __construct()
    {
        $this->institute_name = get_setting_value(Settings::SCHOOL_NAME) ? get_setting_value(Settings::SCHOOL_NAME) : '';
    }
    
    public function index(){
        return view('report.hostel.index');
    }

    public function hostelReport(Request $request){
        $hostel_id = '';
        $hostels = Hostel::pluck('name','id');
        return view('report.hostel.hostel-report', compact('hostel_id','hostels'));
    }

    public function hostelReportSearch(Request $request){
        $request->validate([
            'hostel_id' => 'required',
        ]);

        try {
            // hostel_id
            $hostel_id = $request->hostel_id;
            // Fetch the selected columns or set defaults
            $selectedColumns = $request->input('selected_columns', []);
            if (empty($selectedColumns)) {
                $selectedColumns = ['student_name','student_id','hostel_id'];
            }

            // Always include necessary columns
            $requiredColumns = ['student_id','hostel_id'];
            $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
            $columns = array_diff($selectColumns,['student_name','guardian_contact']);

            // Build the query
            $data = StudentHostel::select($columns)->where('hostel_id',$hostel_id)->with(['studentInfo:id,student_id_no,name,guardian_mobile','hostelRoom.roomType']);

            // Return the DataTable
            return Datatables::of($data)
                ->addColumn('student_name', function ($data) {
                    return $data->studentInfo->name ?? 'N/A'; // Assuming studentInfo relation exists
                })
                ->addColumn('student_id', function ($data) {
                    return $data->studentInfo->student_id_no ?? 'N/A'; // Assuming studentInfo relation exists
                })
                ->addColumn('hostel_room_id', function ($data) {
                    return $data->hostelRoom->room_name ?? 'N/A'; // Assuming hostelRoom relation exists
                })
                ->addColumn('guardian_contact', function ($data) {
                    return $data->studentInfo->guardian_mobile ?? 'N/A'; // Assuming studentInfo relation exists
                })
                ->addColumn('room_type_id', function ($data) {
                    // Check if hostelRoom and roomType are loaded and exist
                    return $data->hostelRoom->roomType->name ?? 'N/A'; // Safely access the roomType name
                })
                ->addColumn('cost', function ($data) {
                    return $data->cost ?? 'N/A';
                })
                ->make(true);
        }catch (\Exception $e) {
            \Log::error('Error in Hostel report: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred. Please try again later.'], 500);
        }
    }

    public function hostelReportPDF(Request $request){
        $request->validate([
            'hostel_id' => 'required',
        ]);
        // hostel_id
        $hostel_id = $request->hostel_id;
        $hostel_name =Hostel::find($hostel_id)->name;
        // Fetch the selected columns or set defaults
        $selectedColumns = $request->input('selected_columns', []);
        if (empty($selectedColumns)) {
            $selectedColumns = ['student_name','student_id','hostel_id'];
        }
        // Always include necessary columns
        $requiredColumns = ['student_id','hostel_id'];
        $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));
        $columns = array_diff($selectColumns,['student_name','guardian_contact']);
        // Build the query
        $data = StudentHostel::select($columns)->where('hostel_id',$hostel_id)->with(['studentInfo:id,student_id_no,name,guardian_mobile','hostelRoom.roomType'])->get();

        $pdf = Pdf::loadView('report.hostel.hostel_pdf', ['hostels'=>$data,'columns'=>$selectedColumns,'hostel_name'=>$hostel_name,'report_name'=>'Hostel Wise Report']);
        return response()->streamDownload(function () use ($pdf) {
            echo $pdf->stream();
        }, 'users.pdf', [
            'Content-Type' => 'application/pdf',
        ]);
    }


    public function hostelReportExcel(Request $request)
    {
        $request->validate([
            'hostel_id' => 'required',
        ]);

        // Fetch hostel details
        $hostel_id = $request->hostel_id;
        $hostel = Hostel::find($hostel_id);

        if (!$hostel) {
            return response()->json(['error' => 'Hostel not found'], 404);
        }

        $hostel_name = $hostel->name;

        // Fetch selected columns or set defaults
        $selectedColumns = $request->input('selected_columns', []);
        if (empty($selectedColumns)) {
            $selectedColumns = ['student_id', 'hostel_room_id'];
        }

        // Always include necessary columns
        $requiredColumns = ['student_id', 'hostel_room_id'];
        $selectColumns = array_unique(array_merge($selectedColumns, $requiredColumns));

        // Avoid fetching unnecessary fields
        $columns = array_diff($selectColumns, ['student_name','guardian_contact']);
        // Fetch data with relationships
        $data = StudentHostel::select($columns)
            ->where('hostel_id', $hostel_id)
            ->with([
                'studentInfo:id,student_id_no,name,guardian_mobile',
                'hostelRoom.roomType',
            ])
            ->get();

        // Prepare metadata for the report
        $metadata = [
            'hostel_name' => $hostel_name,
            'institute_name' => $this->institute_name,
            'report_name' => 'Hostel Wise Report',
        ];

        // Pass data and metadata to the export class
        return Excel::download(new ExportStudentHostel($data, $metadata, $selectedColumns), 'hostel-wise-report.xlsx');
    }



    public function classReport(){
        
    }
    

    
}
