<?php

namespace App\Imports;

use DateTime;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use App\Models\HrPayroll\Employee;
use App\Models\HrPayroll\EmployeeAttendance;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;

class EmployeeAttendanceImport implements ToCollection, WithHeadingRow, WithValidation, SkipsEmptyRows
{
    /**
    * @param Collection $collection
    */
    public function collection(Collection $rows)
    {
        $rows = $rows->toArray();
        $employees = [];

        foreach ($rows as $entry) {
            $timestamp = DateTime::createFromFormat('d/m/Y g:i:s A', $entry['datetime']);
            if (!$timestamp) continue;

            $date = $timestamp->format('Y-m-d');
            $time = $timestamp->format('H:i:s');

            $employees[$entry['no']][$date][] = ['time' => $time, 'raw' => $entry];
        }

        // Process each employee's records
        $results = [];

        foreach ($employees as $emp_no => $dates) {
            foreach ($dates as $date => $punches) {
                // Sort punches by time
                usort($punches, fn($a, $b) => strcmp($a['time'], $b['time']));

                $first_punch = $punches[0]['raw'];
                $last_punch = $punches[count($punches) - 1]['raw'];

                $results[$emp_no][$date] = [
                    'date' => $date,
                    'first_punch' => $first_punch,
                    'last_punch' => $last_punch
                ];
            }
        }
       
        DB::beginTransaction();
        try {

            foreach ($results as $emp_no => $data) {

                $datesArray = array_values($data);
                $employee = Employee::where('employee_code', $emp_no)->first();
                if (!$employee) {
                    continue; // Skip if employee or office details are missing
                }

                foreach($datesArray as $getItem) {

                    // Parse first punch
                    $first_punch_datetime_string = $getItem['first_punch']['datetime'];
                    $first_punch_datetime = Carbon::createFromFormat('d/m/Y h:i:s A', $first_punch_datetime_string);
                    $first_punch_time = $first_punch_datetime->format('H:i:s');
                    $first_punch_date = store_date_format($getItem['date']); // Assuming your function formats the date to Y-m-d

                    // Parse last punch
                    $last_punch_datetime_string = $getItem['last_punch']['datetime'];
                    $last_punch_datetime = Carbon::createFromFormat('d/m/Y h:i:s A', $last_punch_datetime_string);
                    $last_punch_time = $last_punch_datetime->format('H:i:s');
                    // Check if attendance already exists
                    $attendanceExist = EmployeeAttendance::where('employee_id', $employee->id)
                                        ->whereDate('punch_date', $first_punch_date)
                                        ->first();

                    if (!$attendanceExist) {
                        $store = new EmployeeAttendance();
                        $store->employee_id = $employee->id;
                        $store->user_id = $employee->user_id;
                        $store->branch_id = $employee->branch_id;
                        $store->shift_id = $employee->shift_id;
                        $store->department_id = $employee->department_id;
                        $store->designation_id = $employee->designation_id;
                        $store->employee_type_id = $employee->employee_type_id;
                        $store->punch_in = $first_punch_time;
                        $store->punch_out = $last_punch_time;
                        $store->punch_date = $first_punch_date;
                        $store->day_status = 'p';
                        $store->is_present = 1;
                        $store->is_absent = 0;
                        $store->is_late = 0;
                        $store->is_manual = 0;
                        $store->save();
                    } else {
                        $attendanceExist->punch_in = $first_punch_time;
                        $attendanceExist->punch_out = $last_punch_time;
                        $attendanceExist->save();
                    }
                }

            }
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => $e->getMessage()], 500);
        }
        return redirect()->back()->with('success', 'Saved successfully');
    }

    public function rules(): array
    {
        return [
            // 'employee_id' => 'required|integer|exists:employees,id',
            // 'punch_date' => 'required|date',
            // 'punch_in' => 'nullable|date_format:H:i:s',
            // 'punch_date' => 'nullable|date_format:H:i:s',
        ];
    }

}
