<?php

namespace App\Jobs;

use App\Constants\Status;
use App\Models\Result\Grade;
use App\Models\Result\Exam;
use App\Models\Result\MarkHead;
use App\Models\Academic\AcademicYear;
use App\Models\Result\MarkHeadAssign;
use App\Models\Scopes\BranchFilterScope;
use App\Models\SmsRecord;
use App\Models\Student\StudentInfo;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessExamResultJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $exam_id;
    protected $academic_year;
    protected $branch_id;
    public $timeout = 3600; // 1 hour timeout
    public $tries = 3; // Number of retries if job fails

    public function __construct($exam_id, $academic_year, $branch_id)
    {
        $this->exam_id = $exam_id;
        $this->academic_year = $academic_year;
        $this->branch_id = $branch_id;
    }

    public function handle()
    {
        try {
            // Get students directly using the query
            $students = StudentInfo::withoutGlobalScope(BranchFilterScope::class)
            ->whereHas('examRecords', function ($query) {
                $query->withoutGlobalScope(BranchFilterScope::class)
                    ->where('exam_id', $this->exam_id)
                    ->where('year', $this->academic_year);
            })
            ->with(['examRecords' => function ($query) {
                $query->withoutGlobalScope(BranchFilterScope::class)
                    ->where('exam_id', $this->exam_id)
                    ->where('year', $this->academic_year)
                    ->orderBy('subject_id');
            }])
            ->where('status', '=', Status::ACTIVE)
            ->orderBy('student_class_id', 'ASC')
            ->get();

            $grades = Grade::all();
            $markHeads = MarkHead::all();
            $academicYears = AcademicYear::pluck('year', 'id');
            $exam_details = Exam::findOrFail($this->exam_id);

            foreach ($students->chunk(100) as $chunk) {
                foreach ($chunk as $student) {
                    try {
                        // Calculate results directly here
                        $results = $this->calculateResults(
                            $student, 
                            $this->exam_id, 
                            $this->academic_year, 
                            $grades, 
                            $markHeads
                        );

                        $smsBody = "Dear {$student->first_name} {$student->last_name},\nThe Result of your {$exam_details->exam_name} {$academicYears[$this->academic_year]} is {$results['grade_name']} (GPA {$results['gpa']}).\n\nThanks\nJABALUNNUR \nPowered by: EduEasy";
                        
                        $userInfo = [
                            'user_id' => $student->user_id,
                            'student_id' => $student->id,
                            'branch_id' => $student->branch_id ?? auth()->user()->branch_id,
                            'exam_id' => $this->exam_id ?? null,
                        ];

                        $smsRecord = SmsRecord::where('student_id', $student->id)
                            ->where('type', 'student')
                            ->where('exam_id', $this->exam_id)
                            ->first();
                            
                        if ($smsRecord) {
                            continue;
                        }

                        $mobile = $student->mobile_no;
                        if(empty($mobile)){
                            $mobile = $student->guardian_mobile;
                        }
                        if(empty($mobile)){
                            $mobile = $student->father_mobile;
                        }
                        if(empty($mobile)){
                            $mobile = $student->mother_mobile;
                        }

                        if(empty($mobile)){
                            continue;
                        }

                        SendSmsJob::dispatch($mobile, $smsBody, 'student', $userInfo);

                    } catch (\Exception $e) {
                        Log::error("Failed to process SMS for student {$student->id}: " . $e->getMessage());
                    }
                }
            }

        } catch (\Exception $e) {
            Log::error('Failed to process exam result SMS: ' . $e->getMessage());
            throw $e;
        }
    }

    private function calculateResults($student, $exam_id, $academic_year, $grades, $markHeads)
    {
        static $prev_student_class_id = '';
        static $subject_head_marks = [];

        $grand_total = 0;
        $grand_total_obtain = 0;
        $fail_count = 0;

        $student_class_id = $student->student_class_id ?? "";

        if ($prev_student_class_id != $student_class_id) {
            $assignedMarksHead = MarkHeadAssign::where('exam_id', $exam_id)
                ->where('student_class_id', $student_class_id)
                ->pluck('head_marks', 'subject_id');
            $subject_head_marks = $assignedMarksHead;
        }

        foreach ($student->examRecords as $records) {
            $total_marks_obtain = 0;
            $full_marks = 0;

            $head_wise_marks_record = json_decode($records->head_wise_marks, true);
            $head_marks_data = $subject_head_marks[$records->subject->id] ?? null;
            $head_marks_array = json_decode($head_marks_data, true);

            foreach ($markHeads as $head) {
                if (isset($head_marks_array[$head->id])) {
                    $full_marks += $head_marks_array[$head->id];
                    $total_marks_obtain += $head_wise_marks_record[$head->id] ?? 0;
                }
            }

            $grand_total_obtain += $total_marks_obtain;
            $grand_total += $full_marks;

            $total_percentage = $total_marks_obtain > 0 ? round(($total_marks_obtain * 100) / $full_marks) : 0;
            
            foreach ($grades as $grade) {
                if ($grade->mark_from <= $total_percentage && $grade->mark_to >= $total_percentage) {
                    if ($grade->name == 'F') {
                        $fail_count++;
                    }
                    break;
                }
            }
        }

        $grand_total_percentage = $grand_total_obtain > 0 ? round(($grand_total_obtain * 100) / $grand_total) : 0;
        
        $grade_name = '';
        $gpa = '';
        
        foreach ($grades as $grade) {
            if ($grade->mark_from <= $grand_total_percentage && $grade->mark_to >= $grand_total_percentage) {
                $grade_name = $grade->name;
                $gpa = $grade->gpa;
                break;
            }
        }

        if ($fail_count > 0) {
            $grade_name = 'F';
            $gpa = '0.00';
        }

        $prev_student_class_id = $student_class_id;

        return compact('grade_name', 'gpa', 'grand_total_obtain', 'grand_total');
    }
} 