<?php

namespace App\Models\HrPayroll;

use App\Constants\Status;
use App\Models\Academic\Branch;
use App\Models\Academic\Shift;
use App\Models\User;
use App\Traits\LogsActivityTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Spatie\Activitylog\Traits\LogsActivity;
use App\Models\Scopes\BranchFilterScope;
class Employee extends Model
{
    use HasFactory, LogsActivity, LogsActivityTrait;

    protected $guarded = [];

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function employee_education()
    {
        return $this->hasMany(EmployeeEducation::class, 'employee_id');
    }

    public function salaryGrade()
    {
        return $this->belongsTo(SalaryGrade::class, 'salary_grade_id');
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class, 'branch_id');
    }

    public function shift()
    {
        return $this->belongsTo(Shift::class, 'shift_id');
    }

    public function department()
    {
        return $this->belongsTo(HrDepartment::class, 'department_id');
    }

    public function designation()
    {
        return $this->belongsTo(Designation::class, 'designation_id');
    }

    public function employeeType()
    {
        return $this->belongsTo(EmployeeType::class, 'employee_type_id');
    }

    public function employeeSalaries()
    {
        return $this->hasMany(EmployeeSalary::class, 'employee_id');
    }

    public function activeSalary()
    {
        return $this->hasOne(EmployeeSalary::class, 'employee_id')->where('status', '=', Status::ACTIVE);
    }

    protected function logName(): string
    {
        return 'employee';
    }

    protected static function booted()
    {
        static::addGlobalScope(new BranchFilterScope);
    }
    
    protected static function boot()
    {
        parent::boot();
        static::creating(function ($employee) {
            // Generate a unique employee ID number
            $employee->employee_code = static::generateEmployeeId();
        });
    }
    
    protected static function generateEmployeeId()
    {
        $year = date('y'); // Get the last two digits of the current year
        // Get the last generated employee ID number, skipping the BranchFilterScope
        $lastEmployeeIdNo = static::withoutGlobalScope(BranchFilterScope::class)
            ->orderBy('id', 'desc')
            ->first();
        // If there are no existing employee IDs, start with Eyy1001
        if (!$lastEmployeeIdNo) {
            return 'E' . $year . '1001';
        }
        // Extract the numeric part of the last employee ID number
        $employeeIdNo = (int)substr($lastEmployeeIdNo->employee_code, -4) + 1;
        return 'E' . $year . str_pad($employeeIdNo, 4, '0', STR_PAD_LEFT);
    }  


}
