<?php

namespace App\Services\Academic;

use App\Models\Academic\Group;
use App\Models\Academic\Section;
use App\Models\Academic\Shift;
use App\Models\Academic\StudentClass;
use App\Models\Academic\SubjectsAssign;
use App\Models\HrPayroll\Employee;
use App\Models\Academic\ClassRoutine;
use App\Models\Academic\RoutineDetail;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ClassRoutineService
{

    public function get()
    {
        if(auth()->user()->user_type == 'teacher') {
            $myRoutines = RoutineDetail::where('teacher_id', auth()->user()->id)->pluck('class_routine_id');
            $classRoutines = ClassRoutine::whereIn('id', $myRoutines)->with(['shift', 'studentClass', 'section'])->get();
        }else{
            $classRoutines = ClassRoutine::with(['shift', 'studentClass', 'section'])->get();
        }
        return $classRoutines;
    }

    public function store(Request $request)
    {
        // Begin a database transaction
        return DB::transaction(function () use ($request) {

            // Create the class routine
            $days = $request->days;

            $classRoutine = ClassRoutine::create([
                'branch_id' => auth()->user()->branch_id,
                'shift_id' => $request->shift_id,
                'student_class_id' => $request->student_class_id,
                'section_id' => $request->section_id,
                'academic_year_id' => getDefaultAcademicYearID(),
            ]);
    
            // Loop through each day and save the routine details
            foreach ($days as $day) {
                if (isset($request->subjects[strtolower($day)])) {
                    foreach ($request->subjects[strtolower($day)] as $subjectData) {

                        $RoutineDetail[] = [
                            'class_routine_id' => $classRoutine->id,
                            'day' => $day,
                            'subject_id' => $subjectData['subject'],
                            'teacher_id' => $subjectData['teacher'],
                            'start_time' => $subjectData['start_time'],
                            'end_time' => $subjectData['end_time'],
                            'class_room_id' => $subjectData['room_no'],
                            'created_at' => now()
                        ];
                    }
                }
            }
    
            RoutineDetail::insert($RoutineDetail);
        });
    }

    public function getSingleClassRoutine($id)
    {
        return ClassRoutine::with('routineDetails')->find($id);
    }

    public function update(Request $request, ClassRoutine $classRoutine)
    {
        // Begin a database transaction
        return DB::transaction(function () use ($request, $classRoutine) {
            // Update the class routine
            $classRoutine->update([
                'shift_id' => $request->shift_id,
                'student_class_id' => $request->student_class_id,
                'section_id' => $request->section_id,
                'academic_year_id' => getDefaultAcademicYearID(),
            ]);
    
            // Get the days from the request
            $days = $request->days;
    
            // Loop through each day and update routine details
            foreach ($days as $day) {
                // Check if there are subjects for the current day
                if (isset($request->subjects[strtolower($day)])) {
                    foreach ($request->subjects[strtolower($day)] as $index => $subjectData) {
                        // Check if the routine detail exists for this class routine
                        $routineDetail = $classRoutine->routineDetails()
                            ->where('day', $day)
                            ->where('subject_id', $subjectData['subject']) // Adjusting to find by subject_id instead
                            ->first();
    
                        if ($routineDetail) {
                            // Update existing routine detail
                            $routineDetail->update([
                                'class_room_id' => $subjectData['room_no'],
                                'teacher_id' => $subjectData['teacher'],
                                'subject_id' => $subjectData['subject'],
                                'start_time' => $subjectData['start_time'],
                                'end_time' => $subjectData['end_time'],
                            ]);
                        } else {
                            // Create new routine detail if it doesn't exist
                            RoutineDetail::create([
                                'class_routine_id' => $classRoutine->id,
                                'day' => $day,
                                'class_room_id' => $subjectData['room_no'],
                                'teacher_id' => $subjectData['teacher'],
                                'subject_id' => $subjectData['subject'],
                                'start_time' => $subjectData['start_time'],
                                'end_time' => $subjectData['end_time'],
                            ]);
                        }
                    }
                }
            }
        });
    }

    public function getRelatedData()
    {
        return [
            'studentClasses' => StudentClass::pluck('class_name', 'id'),
            'groups' => Group::pluck('group_name', 'id'),
            'shifts' => Shift::pluck('shift_name', 'id'),
            'sections' => Section::pluck('section_name', 'id'),
            'teachers' => User::where('user_type', 'teacher')->pluck('name', 'id'),
        ];
    }

    public function getAssignedSubjects($student_class_id)
    {
        $assigned_subbjects = SubjectsAssign::with('subject')->where('student_class_id', $student_class_id)->get();
        return $assigned_subbjects;
    }

}
