<?php

namespace App\Services\Accounting;

use Carbon\Carbon;
use App\Constants\Status;
use Illuminate\Http\Request;
use App\Filters\InvoiceFilter;
use App\Models\Accounting\FeeType;
use App\Models\Accounting\Invoice;
use App\Models\Student\StudentInfo;
use App\Models\Academic\AcademicYear;
use App\Models\Accounting\StudentFee;
use App\Models\Accounting\InvoiceItem;

class FeeCollectionService
{

    public function getInvoiceData(array $queryParams = [])
    {
        $queryBuilder =  Invoice::with(['student', 'student.studentClass'])->orderBy('id', 'DESC')->latest();

        $students = resolve(InvoiceFilter::class)->getResults([
            'builder' => $queryBuilder,
            'params' => $queryParams
        ]);

        return $students->get();
    }
    
    public function collectFee(Request $request)
    {

        $student = StudentInfo::find($request->student_id);
        $special_discount = $request->special_discount ?? 0;
        $branch_id = auth()->user()->branch_id;

        $totalDiscount = ($special_discount > 0) ? $special_discount : ($request->total_discount ?? null);

        $invoiceData = [
            'student_info_id' => $request->student_id,
            'academic_year_id' => $request->year_id,
            'branch_id' => auth()->user()->branch_id,
            'shift_id' => $student->shift_id,
            'student_class_id' => $student->student_class_id,
            'section_id' => $student->section_id,
            'total_amount' => $request->total_amount,
            'total_discount' => $totalDiscount ?? NULL,
            'late_fine' => $request->late_fine ?? '0.00',
            'paid_amount' => $request->grand_total,
            'payment_date' => date('Y-m-d'),
            'invoice_date' => date('Y-m-d'),
            'due_date' => date('Y-m-d'),
            'payment_method' => $request->payment_method ?? NULL,
            'note' => $request->note ?? NULL,
            'created_by' => auth()->user()->id,
            'status' => Status::PAID
        ];

        $invoice_data = Invoice::create($invoiceData);
        
        $total_amount = 0;
        $total_discount = 0;
        $paid_amount = 0;


        $invoiceItemData = [];

        $feeGroupedByMonth = [];

        foreach ($request->amount as $key => $amount) {
            [$fee_type_id, $month] = explode('_', $key);

            if ($amount) {
                $start_date = Carbon::create(null, $month, 1)->startOfMonth()->toDateString();
                $end_date = Carbon::create(null, $month, 1)->endOfMonth()->toDateString();

                // Grouping fee items by month
                $feeGroupedByMonth[$month]['total_amount'] = ($feeGroupedByMonth[$month]['total_amount'] ?? 0) + $amount;
                $feeGroupedByMonth[$month]['start_date'] = $start_date;
                $feeGroupedByMonth[$month]['end_date'] = $end_date;
                $feeGroupedByMonth[$month]['items'][] = [
                    'fee_type_id' => $fee_type_id,
                    'amount' => $amount,
                ];
            }
        }

        // Insert StudentFee and related InvoiceItem
        foreach ($feeGroupedByMonth as $month => $data) {
            $discount_amount = ($special_discount > 0) ? $special_discount : $this->calculateDiscount($data['total_amount'], $request->discount_type, $request->discount);

            // Create one StudentFee per month and get ID
            $studentFee = StudentFee::create([
                'invoice_id' => $invoice_data->id,
                'branch_id' => $branch_id,
                'academic_year_id' => $request->year_id,
                'student_class_id' => $student->student_class_id,
                'student_id' => $student->id,
                'total_amount' => $data['total_amount'],
                'total_discount' => $discount_amount,
                'grand_total' => $data['total_amount'] - $discount_amount,
                'month' => $month,
                'start_date' => $data['start_date'],
                'end_date' => $data['end_date'],
                'status' => Status::ACTIVE,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $total_amount += $data['total_amount'];
            $total_discount += $discount_amount;
            $paid_amount += $data['total_amount'] - $discount_amount;

            // Create related invoice items with student_fee_id
            foreach ($data['items'] as $item) {
                $invoiceItemData[] = [
                    'invoice_id' => $invoice_data->id,
                    'student_id' => $student->id,
                    'branch_id' => $branch_id,
                    'fee_type_id' => $item['fee_type_id'],
                    'amount' => $item['amount'],
                    'month' => $month,
                    'academic_year_id' => $request->year_id,
                    'student_fee_id' => $studentFee->id, //  Set StudentFee ID here
                    'payment_date' => date('Y-m-d'),
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
        }
        // Insert all invoice items
        InvoiceItem::insert($invoiceItemData);
        $invoice_data->update([
            'total_amount' => $total_amount,
            'total_discount' => $total_discount,
            'late_fine' => lateFineCalculation(),
            'paid_amount' => $paid_amount,

        ]);
        return $invoice_data;
    }

    private function calculateDiscount($amount, $discount_type, $discount)
    {
        if ($discount_type == 'percentage') {
            return ($amount * $discount) / 100;
        } elseif ($discount_type == 'fixedamount') {
            return $discount;
        }
        return 0;
    }

    public function delete($invoice_id)
    {
        Invoice::where('id', $invoice_id)->delete();
        InvoiceItem::where('invoice_id', $invoice_id)->delete();
    }

    public function getSingleInvoiceData($invoiceUuid)
    {
        
        $invoice = Invoice::where('uuid', $invoiceUuid)->firstOrFail();
        $invoice_id = $invoice->id;
        $student_id = $invoice->student_info_id;

        $student_details = StudentInfo::with(['shift', 'studentClass', 'section', 'group'])
            ->where('id', $student_id)
            ->firstOrFail();
        $invoice_items = InvoiceItem::where('invoice_id', $invoice_id)->get();
        $month_invoice_items = $invoice_items->groupBy('month');
        $feeTypes = FeeType::pluck('title', 'id');
        $years = AcademicYear::all();
        $printPreview = "yes";
        return compact('student_details', 'invoice', 'invoice_items', 'month_invoice_items', 'feeTypes', 'years', 'printPreview');
    }

}
