<?php

namespace App\Services\Dashboard;

use Carbon\Carbon;
use App\Constants\Status;
use App\Models\Accounting\Income;
use App\Models\Homework\Homework;
use App\Models\Accounting\Expense;
use App\Models\Accounting\Invoice;
use App\Models\HrPayroll\Employee;
use Illuminate\Support\Facades\DB;
use App\Models\Student\StudentInfo;
use App\Models\Hostel\StudentHostel;
use App\Models\Academic\AcademicYear;
use App\Models\Academic\ClassSection;
use Illuminate\Support\Facades\Cache;
use App\Models\Homework\HomeworkSubmission;
use App\Models\Transport\StudentTransport;

class DashboardService
{

    public function getTotalStudents()
    {
        if(auth()->user()->user_type == 'teacher') {
            $myClass = ClassSection::where('class_teacher_id', auth()->user()->id)->first();
        
            if(!empty($myClass)) {
                $classIds = $myClass->student_class_id;
                $sectionId = $myClass->section_id;
                $shiftId = $myClass->shift_id;
                
                $students = StudentInfo::with(['branch', 'shift', 'section', 'studentClass'])
                    ->where('student_class_id', $classIds)
                    ->where('section_id', $sectionId)
                    ->when($shiftId, function($query) use ($shiftId){
                        return $query->where('shift_id', $shiftId);
                    })
                    ->latest()->get();

                $data['students'] = $students->take(10);
                $data['total_students'] = $students->count();
            } else {
                $data['students'] = collect([]);
                $data['total_students'] = 0;
            }

        } else {
            $data['students'] = StudentInfo::with(['branch', 'shift', 'section', 'studentClass'])->latest()->get();
            $data['total_students'] = $data['students']->count();
            
        }
        
        return $data;
    }

    //total homework
    public function getTotalHomework()
    {
        $homework = Homework::where('teacher_id', auth()->user()->id)->get();
        // Check if homework exists before proceeding
        if ($homework->isEmpty()) {
            return 0; // Return 0 if no homework found
        }
        $classIds = $homework->pluck('student_class_id')->unique();
        $sectionIds = $homework->pluck('section_id')->unique();
        $shiftIds = $homework->pluck('shift_id')->unique();

        $homeworkCount = StudentInfo::whereIn('student_class_id', $classIds)
            ->whereIn('section_id', $sectionIds)
            ->when($shiftIds->isNotEmpty(), function($query) use ($shiftIds){
                return $query->whereIn('shift_id', $shiftIds);
            })
            ->count();
        return $homeworkCount;
    }
    //submitted homework
    public function getSubmittedHomework()
    {
        $submittedHomework = HomeworkSubmission::where('teacher_id', '=',auth()->user()->id)->count();
        return $submittedHomework ?? 0;
    }


    public function getMyStudentAttendance()
    {
        $teacherClass = ClassSection::where('class_teacher_id', auth()->user()->id)->first();
        $data['teacherClass'] = $teacherClass;
        $data['section_id'] = $teacherClass?->section_id;
        $data['student_class_id'] = $teacherClass?->student_class_id;
    

        if(!empty($teacherClass)) {
            $students = StudentInfo::where('student_class_id', $data['student_class_id'])
                ->where('section_id', $data['section_id'])
                ->when($teacherClass?->shift_id, function($query) use ($teacherClass){
                    return $query->where('shift_id', $teacherClass->shift_id);
                })
                ->get();

            $data['attendance_date'] = date('Y-m-d');
            $yesterday = Carbon::yesterday()->toDateString();

            $data['yesterday_present'] = StudentInfo::where('student_class_id', $teacherClass->student_class_id)
            ->where('section_id', $teacherClass->section_id)
            ->when($teacherClass->shift_id, function ($query) use ($teacherClass) {
                $query->where('shift_id', $teacherClass->shift_id);
            })
            ->whereHas('attendance', function ($query) use ($yesterday) {
                $query->whereDate('punch_date', $yesterday);
                $query->where('is_present', 1);
            })
            ->count();

            // Retrieve the attendance records for the specified date
            if (!empty($data['attendance_date'])) {
                $date = date_convertion($data['attendance_date'],'Y-m-d');
                $attendanceRecords = $students->load(['attendance' => function ($query) use ($date) {
                    $query->whereDate('punch_date', $date);
                }]);
                $data['attendanceRecords']  = $attendanceRecords;
                return $data;
            }

        }else{
            return [];
        }
    }

    public function getTotalTeacher()
    {
        return Cache::remember('total_teachers', 60, function () {
            return Employee::where('user_type', 'teacher')->count();
        });
    }

    public function getTotalEmployee()
    {
        return Cache::remember('total_employees', 60, function () {
            return Employee::where('user_type', '=', 'staff')->orWhere('user_type', 'teacher')->count();
        });
    }

    public function getTodaysCollection()
    {
        $today = Carbon::now()->toDateString();
    
        $collection = Invoice::whereDate('invoice_date', $today)
            ->where('status', Status::PAID)
            ->whereNotNull('payment_date')
            ->sum('paid_amount');
        return $collection;
    }

    public function getThisMonthCollection(){
        $start_month = Carbon::now()->format('Y-m-01');
        $end_month = Carbon::now()->format('Y-m-t');
        $collection = Invoice::whereBetween('invoice_date', [$start_month, $end_month])
            ->where('status', Status::PAID)
            ->whereNotNull('payment_date')
            ->sum('paid_amount');
        return $collection;
    }

    public function getThisMonthIncome(){
        $start_month = Carbon::now()->format('Y-m-01');
        $end_month = Carbon::now()->format('Y-m-t');
        $income = Income::whereBetween('income_date', [$start_month, $end_month])->sum('amount');
        return $income;
    }

    public function getThisMonthExpense(){
        $start_month = Carbon::now()->format('Y-m-01');
        $end_month = Carbon::now()->format('Y-m-t');
        $expense = Expense::whereBetween('expense_date', [$start_month, $end_month])->sum('amount');
        return $expense;
    }

    //this month due
    public function getThisMonthDue()
    {
        $branch_id = auth()->user()->branch_id;
        $academicYear = getDefaultAcademicYearValue2();
        $academicYearId = getDefaultAcademicYearId2();
        $currentMonth = now()->format('n');
        $lastDayOfMonth = Carbon::now()->endOfMonth()->format('d');
        $fromDate = $academicYear . '-' . $currentMonth . '-01';
        $toDate = $academicYear . '-' . $currentMonth . '-' . $lastDayOfMonth;


        $academicYear = getDefaultAcademicYearValue2();
     
        $branch_id = auth()->user()->branch_id;
        $year_id = AcademicYear::where('year', $academicYear)->value('id');

        $months =  range($currentMonth, $currentMonth);

        $students = DB::table('student_infos as si')
            ->join('student_classes as sc', 'si.student_class_id', '=', 'sc.id')
            ->where('si.branch_id', $branch_id)
            ->select([
                'si.id as student_id',
                'si.name',
                'si.student_id_no',
                'sc.class_name',
                'si.student_class_id'
            ])
            ->get();

        $results = collect();

        foreach ($students as $student) {
            foreach ($months as $month) {

                $monthlyHostelCost = StudentHostel::where('student_id', $student->student_id)->first()->cost ?? 0 ;
                $monthlyTransportCost = StudentTransport::where('student_id', $student->student_id)->first()->monthly_cost ?? 0 ;
                $classFees = DB::table('class_fees')
                    ->where('branch_id', $branch_id)
                    ->where('academic_year_id', $year_id)
                    ->where('student_class_id', $student->student_class_id)
                    ->where('month', $month)
                    ->sum('amount');
                // Sum everything together for the total
                $total = $classFees + $monthlyHostelCost + $monthlyTransportCost;

                $paid = DB::table('student_fees')
                    ->where('branch_id', $branch_id)
                    ->where('academic_year_id', $year_id)
                    ->where('student_id', $student->student_id)
                    ->where('month', $month)
                    ->sum('total_amount');

                // Show row if any fee exists or any amount was paid
                if ($total == 0 && $paid == 0) {
                    continue;
                }

                $results->push((object)[
                    'student_id' => $student->student_id,
                    'name' => $student->name,
                    'student_id_no' => $student->student_id_no,
                    'class_name' => $student->class_name,
                    'month' => $month,
                    'total_amount' => number_format($total, 2, '.', ''),
                    'paid_amount' => number_format($paid, 2, '.', ''),
                ]);
            }
        }

        $grandTotalDue = $results->sum(function ($item) {
            return (float) $item->total_amount - (float) $item->paid_amount;
        });

        $grandTotalDue = number_format($grandTotalDue, 2); // e.g., "1090.00"

        return $grandTotalDue ?? 0;
    }
    

    public function getMonthWiseIncome()
    {
        $currentYear = date('Y');

        $monthWiseIncome = DB::table('incomes')
            ->select(
                DB::raw('MONTH(income_date) as month'),
                DB::raw('SUM(amount) as total_income')
            )
            ->whereYear('income_date', $currentYear)
            ->groupBy(DB::raw('MONTH(income_date)'))
            ->orderBy(DB::raw('MONTH(income_date)'))
            ->where('branch_id', auth()->user()->branch_id)
            ->get()
            ->keyBy('month');
            
        $chartData = [];
        for ($month = 1; $month <= 12; $month++) {
            $chartData[] = $monthWiseIncome->get($month)->total_income ?? 0;
        }
 
        // Prepare data for Chart.js
        return $chartData;
    }

}