<?php

namespace App\Services\ExamResult;

use App\Models\Academic\StudentClass;
use App\Models\Academic\Group;
use App\Models\Academic\Shift;
use App\Models\Academic\Section;
use App\Models\Result\MarkHead;
use App\Models\Academic\AcademicYear;
use App\Models\Academic\SubjectsAssign;
use App\Models\Result\Exam;
use App\Models\Result\ExamRecord;
use App\Models\Result\Grade;
use App\Models\Result\MarkHeadAssign;
use App\Models\Student\StudentInfo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;


class MarksEntryService
{
    public function getRelatedData()
    {
        return [
            'studentClasses' => StudentClass::pluck('class_name','id'),
            'groups' => Group::pluck('group_name', 'id'),
            'sections' => Section::pluck('section_name','id'),
            'grades' => Grade::orderBy('mark_from', 'desc')->get(),
            'markHeads' => MarkHead::all(),
            'academicYears' => AcademicYear::orderBy('year','DESC')->pluck('year','id'),
            'exams' => Exam::orderBy('id', 'desc')->select('id', 'exam_name', 'year')->get(),
            'shifts' => Shift::select('id', 'shift_name')->get(),
        ];
    }

    public function getMarksSheetByClass(Request $request)
    {
        if(!$request->all()){
            return [[], [], [], '', []];
        }
        // dd($request->all());
        $exam_id = $request->exam_id;
        $exam = Exam::where('id', $exam_id)->first();
        $academic_year = $exam->year;
        // dd($academic_year);
        $shift_id = $request->shift_id;
        
        $section_id = $request->section_id;
        $group_id = $request->group_id;
        $student_class_id = $request->student_class_id;

        $assignedMarksHead = MarkHeadAssign::where('exam_id', $exam_id)
            ->where('student_class_id', $student_class_id)
            ->pluck('head_marks', 'subject_id');

        $subjects_assigns = SubjectsAssign::where('student_class_id',$student_class_id)->select('id','student_class_id','subject_id')->get();
        // Start with exam records instead of students
        $examRecords = ExamRecord::where('exam_id', $exam_id)
            ->where('year', $academic_year)
            ->where('class_id', $student_class_id)
            ->when(!empty($shift_id), function ($query) use ($shift_id) {
                $query->where('shift_id', $shift_id);
            })
            ->when(!empty($section_id), function ($query) use ($section_id) {
                $query->where('section_id', $section_id);
            })
            ->when(!empty($group_id), function ($query) use ($group_id) {
                $query->where('group_id', $group_id);
            });

        // Get unique student IDs from exam records
        $studentIds = $examRecords->pluck('student_id')->unique();

        // Get students with their exam records
        // $students = StudentInfo::whereIn('id', $studentIds)
        //     ->with(['examRecords' => function ($query) use ($exam_id, $academic_year, $student_class_id) {
        //         $query->where('exam_id', $exam_id)
        //             ->where('year', $academic_year)
        //             ->where('class_id', $student_class_id)
        //             ->orderBy('subject_id');
        //     }, 'examRecords.subject'])
        //     ->orderBy('class_roll', 'ASC')
        //     ->get();

        $students = StudentInfo::whereIn('id', $studentIds)
            ->when(!empty($group_id), function ($query) use ($group_id, $student_class_id) {
                $query->where('group_id', $group_id)
                    ->whereHas('StudentInfoSubjectClass', function ($q) use ($student_class_id) {
                        
                        if (!empty($student_class_id)) {
                            $q->where('student_class_id', $student_class_id);
                        }
                    });
            })
            ->with([
                'examRecords' => function ($query) use ($exam_id, $academic_year, $student_class_id) {
                    $query->where('exam_id', $exam_id)
                        ->where('year', $academic_year)
                        ->where('class_id', $student_class_id)
                        ->orderBy('subject_id');
                },
                'examRecords.subject',
                'StudentInfoSubjectClass' => function ($query) {
                    $query->orderBy('subject_id', 'asc');
                }
            ])
            ->orderBy('class_roll', 'ASC')
            ->get();

            // dd($students);

        // Calculate highest marks
        $highestMarks = ExamRecord::where('class_id', $student_class_id)
            ->where('exam_id', $exam_id)
            ->where('year', $academic_year)
            ->when(!empty($shift_id), function ($query) use ($shift_id) {
                $query->where('shift_id', $shift_id);
            })
            ->when(!empty($section_id), function ($query) use ($section_id) {
                $query->where('section_id', $section_id);
            })
            ->when(!empty($group_id), function ($query) use ($group_id) {
                $query->where('group_id', $group_id);
            })
            ->groupBy('subject_id')
            ->select('subject_id', DB::raw('MAX(total_marks) as highest_marks'))
            ->get();

        $highest_marks_array = $highestMarks->pluck('highest_marks', 'subject_id')->toArray();

        $publish_date = Exam::where('id', $exam_id)->value('publish_date');
            // dd($students, $highest_marks_array, $assignedMarksHead, $publish_date, $subjects_assigns, "mark Sheet");
        return [$students, $highest_marks_array, $assignedMarksHead, $publish_date, $subjects_assigns];
    }

    public function AllMarkSheetDownload(Request $request)
    {
        
        // dd($request->all());
        if(!$request->all()){
            return [[], [], [], '', []];
        }
        // $exam_id = $request->exam_id;
        // $exam = Exam::where('id', $exam_id)->first();
        // $academic_year = $exam->year;
        // $shift_id = $request->shift_id;
        // $section_id = $request->section_id ?? null;
        // $group_id = $request->group_id ?? null;
        // $student_class_id = $request->student_class_id;
        
        $exam_id = (int) $request->exam_id;
        $exam = Exam::where('id', $exam_id)->first();
        $academic_year = $exam->year;
        $shift_id = (int) $request->shift_id;
        $section_id = $request->section_id !== null ? (int) $request->section_id : null;
        $group_id = $request->group_id !== null ? (int) $request->group_id : null;
        $student_class_id = (int) $request->student_class_id;

        
        // dd($request->all(), $exam_id, $academic_year, $shift_id, $section_id, $group_id, $student_class_id);
        $assignedMarksHead = MarkHeadAssign::where('exam_id', $exam_id)
            ->where('student_class_id', $student_class_id)
            ->pluck('head_marks', 'subject_id');

        $subjects_assigns = SubjectsAssign::where('student_class_id',$student_class_id)->select('id','student_class_id','subject_id')->get();
        // Start with exam records instead of students
        $examRecords = ExamRecord::where('exam_id', $exam_id)
            ->where('year', $academic_year)
            ->where('class_id', $student_class_id)
            ->when(!empty($shift_id), function ($query) use ($shift_id) {
                $query->where('shift_id', $shift_id);
            })
            ->when(!empty($section_id), function ($query) use ($section_id) {
                $query->where('section_id', $section_id);
            })
            ->when(!empty($group_id), function ($query) use ($group_id) {
                $query->where('group_id', $group_id);
            })->get();
        

        // Get unique student IDs from exam records
        $studentIds = $examRecords->pluck('student_id')->unique();
        // dd($studentIds);
        // Get students with their exam records
        $students = StudentInfo::whereIn('id', $studentIds)->where('status','active')
            ->with(['examRecords' => function ($query) use ($exam_id, $academic_year, $student_class_id) {
                $query->where('exam_id', $exam_id)
                    ->where('year', $academic_year)
                    ->where('class_id', $student_class_id)
                    ->orderBy('subject_id');
            }, 'examRecords.subject'])
            ->orderBy('class_roll', 'ASC')
            ->get();

        // Calculate highest marks
        $highestMarks = ExamRecord::where('class_id', $student_class_id)
            ->where('exam_id', $exam_id)
            ->where('year', $academic_year)
            ->when(!empty($shift_id), function ($query) use ($shift_id) {
                $query->where('shift_id', $shift_id);
            })
            ->when(!empty($section_id), function ($query) use ($section_id) {
                $query->where('section_id', $section_id);
            })
            ->when(!empty($group_id), function ($query) use ($group_id) {
                $query->where('group_id', $group_id);
            })
            ->groupBy('subject_id')
            ->select('subject_id', DB::raw('MAX(total_marks) as highest_marks'))
            ->get();

        $highest_marks_array = $highestMarks->pluck('highest_marks', 'subject_id')->toArray();

        $publish_date = Exam::where('id', $exam_id)->value('publish_date');
        // dd($highest_marks_array);
            // dd($students, $highest_marks_array, $assignedMarksHead, $publish_date, $subjects_assigns, "mark Sheet");
        return [$students, $highest_marks_array, $assignedMarksHead, $publish_date, $subjects_assigns];
    }



    public function individuleMarkSheetDownload(Request $request)
    {
        // dd($request->all());
        $student_info = StudentInfo::where('student_id_no', $request->student_id)->first();
        if(!$request->all()){
            return [[], [], [], '', []];
        }
        $exam_id = $request->exam_id;
        $exam = Exam::where('id', $exam_id)->first();
        $academic_year = $exam->year;
        $shift_id = $request->shift_id;
        $section_id = $request->section_id ?? null;
        $group_id = $request->group_id ?? null;
        $student_class_id = $request->student_class_id;
        // dd($request->all());
        $subjects_assigns = SubjectsAssign::where('student_class_id',$student_class_id)->select('id','student_class_id','subject_id')->get();
        $assignedMarksHead = MarkHeadAssign::where('exam_id', $exam_id)
            ->where('student_class_id', $student_class_id)
            ->pluck('head_marks', 'subject_id');
        // Start with exam records instead of students
        $examRecords = ExamRecord::where('exam_id', $exam_id)
            ->where('student_id', $student_info->id)
            ->where('year', $academic_year)
            ->where('class_id', $student_class_id)
            ->when(!empty($shift_id), function ($query) use ($shift_id) {
                $query->where('shift_id', $shift_id);
            })
            ->when(!empty($section_id), function ($query) use ($section_id) {
                $query->where('section_id', $section_id);
            })
            ->when(!empty($group_id), function ($query) use ($group_id) {
                $query->where('group_id', $group_id);
            });
        // Get unique student IDs from exam records
        $studentIds = $examRecords->pluck('student_id')->unique();
        // Get students with their exam records
        $students = StudentInfo::where('id', $student_info->id)
            ->with(['examRecords' => function ($query) use ($exam_id, $academic_year, $student_class_id) {
                $query->where('exam_id', $exam_id)
                    ->where('year', $academic_year)
                    ->where('class_id', $student_class_id)
                    ->orderBy('subject_id');
            }, 'examRecords.subject'])
            ->orderBy('class_roll', 'ASC')
            ->get();
        // Calculate highest marks
        $highestMarks = ExamRecord::where('class_id', $student_class_id)
            ->where('exam_id', $exam_id)
            ->where('year', $academic_year)
            // ->when(!empty($shift_id), function ($query) use ($shift_id) {
            //     $query->where('shift_id', $shift_id);
            // })
            // ->when(!empty($section_id), function ($query) use ($section_id) {
            //     $query->where('section_id', $section_id);
            // })
            // ->when(!empty($group_id), function ($query) use ($group_id) {
            //     $query->where('group_id', $group_id);
            // })
            ->groupBy('subject_id')
            ->select('subject_id', DB::raw('MAX(total_marks) as highest_marks'))
            ->get();

        $highest_marks_array = $highestMarks->pluck('highest_marks', 'subject_id')->toArray();

        $publish_date = Exam::where('id', $exam_id)->value('publish_date');
        return [$students, $highest_marks_array, $assignedMarksHead, $publish_date, $subjects_assigns];
    }

}
