<?php

namespace App\Services\HrPayroll;

use App\Constants\Accounting;
use App\Constants\Settings;
use App\Constants\Status;
use App\Filters\EmployeeSalaryFilter;
use App\Models\Academic\AcademicYear;
use App\Models\Academic\Shift;
use App\Models\Accounting\AccountingHead;
use App\Models\Accounting\Expense;
use App\Models\HrPayroll\Employee;
use App\Models\HrPayroll\EmployeeSalary;
use App\Models\HrPayroll\MonthlySalary;
use App\Models\HrPayroll\SalaryGrade;
use App\Models\HrPayroll\SalaryHead;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class EmployeeSalaryService
{

    public function getEmployeeSalary(Request $request)
    {
        $queryBuilder = EmployeeSalary::with(['employee', 'salaryGrade.salaryGradeRules'])->orderBy('id', 'desc');
        if (isset($request->from_date)  && isset($request->to_date)) {
            $queryBuilder = $queryBuilder->whereBetween('created_at', [$request->from_date, $request->to_date]);
        }
        if (isset($request->employee_code)) {
            $employee = Employee::where('employees.employee_code', '=', $request->employee_code)->first();
            $queryBuilder = $queryBuilder->where('employee_id', '=', $employee->id);
        }

        return $queryBuilder->get();
    }


    public function salarySheet(Request $request)
    {

        $queryBuilder = MonthlySalary::with('employee', 'employee.branch', 'employee.shift', 'salaryGrade.salaryGradeRules')
            ->where('month', $request->month)
            ->where('academic_year_id', $request->academic_year);

        if ($request->employee_code) {
            $employee = Employee::where('employees.employee_code', '=', $request->employee_code)->first();
            $queryBuilder = $queryBuilder->where('employee_id', '=', $employee->id);
        }

        return $queryBuilder->get();
    }

    public function getEmployeeMonthlySalary(array $queryParams = [])
    {
        // Return early if no query parameters are provided
        if (empty($queryParams)) {
            return '';
        }

        // Build the base query for employees with related data
        $queryBuilder = Employee::with(['activeSalary.salaryGrade.salaryGradeRules', 'branch', 'shift'])
            ->leftJoin('monthly_salaries', function ($join) use ($queryParams) {
                $hasMonthAndYear = isset($queryParams['month'], $queryParams['academic_year']);
                // Conditionally join based on month and academic year filters
                $join->on('employees.id', '=', 'monthly_salaries.employee_id')
                    ->when($hasMonthAndYear, function ($join) use ($queryParams) {
                        $join->where('monthly_salaries.month', '=', $queryParams['month'])
                            ->where('monthly_salaries.academic_year_id', '=', $queryParams['academic_year']);
                    });
            })
            ->select(
                'employees.id',
                'employees.name',
                'employees.employee_code',
                'employees.mobile_number',
                'employees.email',
                'employees.joining_date',
                'employees.created_at',
                DB::raw('COALESCE(monthly_salaries.amount, 0) as paid_amount'),
                DB::raw('CASE WHEN COALESCE(monthly_salaries.amount, 0) = 0 OR monthly_salaries.status = \'' . Status::DUE . '\' THEN \'' . Status::DUE . '\' ELSE \'' . Status::PAID . '\' END as payment_status')
            )
            ->orderBy('employees.name', 'asc');

        // Apply filters using the pipeline
        $filteredEmployees = resolve(EmployeeSalaryFilter::class)->getResults([
            'builder' => $queryBuilder,
            'params' => $queryParams,
        ]);

        return $filteredEmployees->get();
    }


    public function storeEmployeeSalary($request)
    {
        $from_date = date('Y-m-1');
        $to_date = date('Y-m-t');

        if ($request->salary && $request->salary_grade && $request->employee_code) {

            $employee_id = Employee::where('employee_code', $request->employee_code)->first()?->id;

            $employee_salary = EmployeeSalary::where('employee_id', $employee_id)
                ->whereBetween('created_at', [$from_date,  $to_date])
                ->where('salary_grade_id', $request->salary_grade)
                ->where('status', Status::ACTIVE)
                ->first();

            $monthly_salary = MonthlySalary::where('employee_id', $employee_id)
                ->whereBetween('process_date', [$from_date,  $to_date])
                ->where('salary_grade_id', $request->salary_grade)
                ->where('status', Status::PAID)->first();

            if (!empty($employee_salary) && empty($monthly_salary)) {
                //update employee salary
                $employee_salary->update([
                    'employee_id' => $employee_id,
                    'branch_id' => auth()->user()->branch_id,
                    'status' => Status::ACTIVE,
                    'salary_grade_id' => $request->salary_grade,
                    'amount' => $request->salary,
                ]);
            } else {
                //insert employee salary
                EmployeeSalary::where('employee_id', $employee_id)->update(['status' => Status::INACTIVE]);
                EmployeeSalary::create([
                    'employee_id' => $employee_id,
                    'branch_id' => auth()->user()->branch_id,
                    'status' => Status::ACTIVE,
                    'salary_grade_id' => $request->salary_grade,
                    'amount' => $request->salary,
                ]);
            }

            Employee::where('id', $employee_id)->update(['salary' => $request->salary]);
        }
    }


    public function storeMonthlySalary(Request $request)
    {
        $validatedData = $request->validate([
            'selected_employees' => 'required|array',
            'month' => 'required|numeric',
            'academic_year' => 'required|numeric',
        ]);

        $selectedEmployees = $validatedData['selected_employees'];
        $month = $validatedData['month'];
        $academicYear = $validatedData['academic_year'];
        $employee_code = $request->employee_code;

        $total_salary_amount = 0;

        // Loop through selected employees
        foreach ($selectedEmployees as $employeeId) {

            $employee = Employee::findOrFail($employeeId);

            // Check if salary already exists for the given employee, month, and academic year
            $existingSalary = MonthlySalary::where('employee_id', $employeeId)
                ->where('month', $month)
                ->where('academic_year_id', $academicYear)
                ->first();

            if ($existingSalary) {
                // If salary exists, here we skip
                continue;
            }

            // Get the active salary for the employee
            $activeSalary = $employee->activeSalary;

            $total_salary_amount += $activeSalary->amount;

            $salaruData[] = [
                'employee_id' => $employeeId,
                'branch_id' => auth()->user()->branch_id,
                'month' => $month,
                'academic_year_id' => $academicYear,
                'process_date' => date('Y-m-d'),
                'salary_grade_id' => $activeSalary->salary_grade_id,
                'amount' => $activeSalary->amount,
                'given_by' => auth()->user()->id,
                'status' => Status::PAID,
                'created_at' => now()
            ];
        }

        if (!empty($salaruData)) {
            MonthlySalary::insert($salaruData);
        }

        $salary_expense_head = get_setting_value(Settings::SALARY_EXPENSE_HEAD);
        if ($salary_expense_head && $total_salary_amount > 0) {
            $this->createSalaryExpense($academicYear, $month, $salary_expense_head, $total_salary_amount);
        }

        $newSalaries = MonthlySalary::with('employee', 'employee.branch', 'employee.shift')
            ->whereIn('employee_id', $selectedEmployees)
            ->where('month', $month)
            ->where('academic_year_id', $academicYear)
            ->get();

        return [
            'newSalaries' => $newSalaries,
            'academicYear' => $academicYear,
            'month' => $month,
            'employee_code' => $employee_code
        ];
    }

    public function createSalaryExpense($year_id, $month, $salary_expense_head, $total_salary_amount)
    {

        $year = AcademicYear::pluck('year', 'id')->toArray()[$year_id];
        $month = Carbon::create()->month($month)->format('F');;
        $title = "Salary of $month $year";

        Expense::create([
            'title' => $title,
            'accounting_head_id' => $salary_expense_head,
            'expense_date' => date('Y-m-d'),
            'amount' => $total_salary_amount,
            'payment_method' => Accounting::CASH,
        ]);
    }

    public function monthlySalarySheet(Request $request)
    {

        $newSalaries = MonthlySalary::with('employee', 'employee.branch', 'employee.shift', 'salaryGrade.salaryGradeRules')
            ->where('month', $request->month)
            ->where('academic_year_id', $request->academicYear)
            ->get();

        return [
            'newSalaries' => $newSalaries,
            'academicYear' => $request->academicYear,
            'month' => $request->month,
            'salary_heads' => SalaryHead::get(),
        ];
    }

    public function getEmployeeByCode($employeeCode)
    {
        if ($employeeCode) {
            return Employee::with(['employeeSalaries', 'salaryGrade', 'branch', 'shift'])->where('employee_code', '=', $employeeCode)->first();
        } else {
            return '';
        }
    }

    public function getRelatedData()
    {
        return [
            'academicYears' => AcademicYear::pluck('year', 'id'),
            'shifts' => Shift::pluck('shift_name', 'id'),
            'salary_grades' => SalaryGrade::pluck('title', 'id'),
            'salary_heads' => SalaryHead::get(),
        ];
    }
}
