<?php

namespace App\Services\Inventory;

use App\Models\Inventory\TransactionMaster;
use Illuminate\Support\Facades\Log;
use App\Models\Student\StudentInfo;
use App\Models\Inventory\StockOut;
use Illuminate\Support\Facades\DB;
use App\Models\Inventory\Product;
use Illuminate\Http\Request;
use Carbon\Carbon;

class SalesService
{

    public function branchWiseTodaySale($branch_id){
        $today = Carbon::today()->toDateString();
        return StockOut::join('transaction_master', 'stock_out.transaction_id', '=', 'transaction_master.id')
            ->where('transaction_master.branch_id', $branch_id)
            ->whereDate('transaction_master.transaction_date', $today)
            ->select(
                'transaction_master.branch_id',
                'stock_out.accounting_head_id',
                DB::raw('SUM(stock_out.final_amount) as total_amount')
            )
            ->groupBy('transaction_master.branch_id', 'stock_out.accounting_head_id')
            ->get();
    }

    // public function branchWiseTodaySale(){
    //     $today = Carbon::today()->toDateString();
    //     return TransactionMaster::whereDate('transaction_date', $today)
    //         ->select('branch_id', DB::raw('SUM(grand_total) as branch_total'))
    //         ->groupBy('branch_id')
    //         ->get();
    // }
    
    public function getAllSales()
    {
        return TransactionMaster::where('transaction_type', 'sale')
            ->where('branch_id', auth()->user()->branch_id)
            ->with('stockOut')
            ->orderBy('id', 'DESC')
            ->get();
    }

    public function getSaleById($id)
    {
        return TransactionMaster::with('stockOut')->findOrFail($id);
    }

    public function createSale(Request $request)
    {
        try {
            $request->validate([
                'student_id_no' => 'required',
                'product_id.*' => 'required',
                'quantity.*' => 'required',
                'price.*' => 'required',
                'total_amount' => 'required',
                'total_discount' => 'required',
                'grand_total' => 'required',
            ]);
            return DB::transaction(function () use ($request) {
                // Create transaction summary
                $transaction = TransactionMaster::create([
                    'branch_id' => auth()->user()->branch_id,
                    'student_id' => $request->student_id_no,
                    'total_amount' => $request->total_amount,
                    'total_discount' => $request->total_discount,
                    'grand_total' => $request->grand_total,
                    'transaction_type' => 'sale',
                    'transaction_date' => date('Y-m-d'),
                    'invoice_number' => $this->generateInvoiceCode(),
                ]);

                // Store sales details
                foreach ($request->product_id as $key => $productId) {
                    $product = Product::find($productId);
                    StockOut::create([
                        'transaction_id' => $transaction->id,
                        'product_id' => $productId,
                        'accounting_head_id' => $product->accounting_head_id,
                        'quantity' => $request->quantity[$key],
                        'price' => $request->price[$key],
                        'discount_amount' => $request->discount_amount[$key] ?? 0,
                        'discount_percentage' => $request->discount_percentage[$key] ?? 0,
                        'final_amount' => $request->total[$key] ?? 0,
                    ]);
                }

                return $transaction;
            });
        } catch (\Exception $e) {
            Log::error('Error creating sale: ' . $e->getMessage());
            throw new \Exception('Failed to create sale. Please try again.'); // Or return a custom error response
        }
    }

    public function updateSale(Request $request, $id)
    {
        try {
            return DB::transaction(function () use ($request, $id) {
                $sale = TransactionMaster::findOrFail($id);
                $sale->update([
                    'branch_id' => auth()->user()->branch_id,
                    'student_id' => $request->student_id_no,
                    'total_amount' => $request->total_amount,
                    'total_discount' => $request->total_discount,
                    'grand_total' => $request->grand_total,
                    'transaction_date' => date('Y-m-d'),
                ]);

                $sale->stockOut()->whereNotIn('product_id', $request->product_id)->delete();

                foreach ($request->product_id as $key => $productId) {
                    $product = Product::find($productId);
                    
                    $quantity = $request->quantity[$key];
                    $price = $request->price[$key];
                    $discountAmount = $request->discount_amount[$key] ?? 0;
                    $discountPercentage = $request->discount_percentage[$key] ?? 0;
                    $final_amount = $request->total[$key] ?? 0;

                    $stockOut = StockOut::where('transaction_id', $sale->id)
                        ->where('product_id', $productId)
                        ->first();

                    if ($stockOut) {
                        $stockOut->update([
                            'quantity' => $quantity,
                            'price' => $price,
                            'discount_amount' => $discountAmount,
                            'discount_percentage' => $discountPercentage,
                            'final_amount' => $final_amount,
                        ]);
                    } else {
                        StockOut::create([
                            'transaction_id' => $sale->id,
                            'product_id' => $productId,
                            'accounting_head_id' => $product->accounting_head_id,
                            'quantity' => $quantity,
                            'price' => $price,
                            'discount_amount' => $discountAmount,
                            'discount_percentage' => $discountPercentage,
                            'final_amount' => $final_amount,
                        ]);
                    }
                }

                return $sale;
            });
        } catch (\Exception $e) {
            Log::error('Error updating sale: ' . $e->getMessage());
            throw new \Exception('Failed to update sale. Please try again.'); // Or return a custom error response
        }
    }


    // public function updateSale(Request $request, $id)
    // {
    //     try {
    //         return DB::transaction(function () use ($request, $id) {
    //             $sale = TransactionMaster::findOrFail($id);
    //             $sale->update($request->only(['student_id', 'total_amount']));
    //             foreach ($request->product_id as $key => $productId) {
    //                 $quantity = $request->quantity[$key];
    //                 $price = $request->price[$key];
    //                 $discountAmount = $request->discount_amount[$key] ?? 0;
    //                 $discountPercentage = $request->discount_percentage[$key] ?? 0;
    //                 $stockOut = StockOut::where('transaction_id', $sale->id)
    //                     ->where('product_id', $productId)
    //                     ->first();
    //                 if($stockOut){
    //                     $stockOut->update([
    //                         'quantity' => $quantity,
    //                         'price' => $price,
    //                         'discount_amount' => $discountAmount,
    //                         'discount_percentage' => $discountPercentage,
    //                     ]);
    //                 }else{
    //                     StockOut::create([
    //                         'transaction_id' => $sale->id,
    //                         'product_id' => $productId,
    //                         'quantity' => $quantity,
    //                         'price' => $price,
    //                         'discount_amount' => $discountAmount,
    //                         'discount_percentage' => $discountPercentage,
    //                     ]);
    //                 }
    //             }
    //             return $sale;
    //         });
    //     } catch (\Exception $e) {
    //         Log::error('Error updating sale: ' . $e->getMessage());
    //         throw new \Exception('Failed to update sale. Please try again.'); // Or return a custom error response
    //     }
    // }

    public function deleteSale($id)
    {
        try {
            return DB::transaction(function () use ($id) {
                $sale = TransactionMaster::findOrFail($id);
                $sale->stockOut()->delete();
                $sale->delete();
            });
        } catch (\Exception $e) {
            Log::error('Error deleting sale: ' . $e->getMessage());
            throw new \Exception('Failed to delete sale. Please try again.'); // Or return a custom error response
        }
    }

    public function searchProducts(Request $request)
    {
        $query = $request->input('query');
        return Product::where('name', 'like', '%' . $query . '%')
        ->where('branch_id', auth()->user()->branch_id)
        ->get();
    }

    public function productSearch(Request $request)
    {   
        $query = $request->input('query');
        return Product::where('name', 'like', '%' . $query . '%')
        ->where('branch_id', auth()->user()->branch_id)
        ->first();
    }

    public function generateInvoiceCode()
    {
        $yearPrefix = date('y');

        $lastInvoice = TransactionMaster::where('invoice_number', 'LIKE', "$yearPrefix%")
                                        ->orderBy('invoice_number', 'desc')
                                        ->first();
        if ($lastInvoice) {
            $lastNumber = (int) substr($lastInvoice->invoice_number, 2);
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1000;
        }
        $newInvoiceCode = $yearPrefix . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
        return $newInvoiceCode;
    }

    
}
