<?php

namespace App\Services\Reports;

use App\Models\Academic\AcademicYear;
use App\Models\Academic\Group;
use App\Models\Academic\Section;
use App\Models\Academic\Shift;
use App\Models\Academic\StudentClass;
use App\Models\Accounting\Expense;
use App\Models\Accounting\Income;
use App\Models\Accounting\Invoice;
use App\Models\HrPayroll\MonthlySalary;
use App\Models\HrPayroll\SalaryHead;
use App\Models\Student\StudentInfo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AccountingReportService
{

    public function dailyCollection(Request $request)
    {
        if (empty($request->all())) {
            return collect();
        }

        list($fromDate, $toDate) = parseDateRange($request->date_range);

        return Invoice::where('status', 'paid')
        ->whereBetween('invoice_date', [$fromDate, $toDate])
        ->selectRaw('
            invoice_date,
            SUM(total_amount) as total_amount,
            SUM(total_discount) as total_discount,
            SUM(late_fine) as total_late_fine,
            SUM(paid_amount) as total_paid
        ')
        ->groupBy('invoice_date')
        ->orderBy('invoice_date')
        ->get();
    }


    public function dailyCollectionWithColumns(Request $request)
    {
        $data = $this->dailyCollection($request);
        return $data->map(function ($student) {
            return [
                'invoice_date' => $student->invoice_date,
                'total_amount' => $student->total_amount,
                'total_discount' => $student->total_discount,
                'total_late_fine' => $student->total_late_fine,
                'total_paid' => $student->total_paid,
            ];
        });
    }


    public function paymentDetailsReport(Request $request)
    {
        if (empty($request->all())) {
            return collect();
        }
    
        $student_id_no = $request->student_id_no;
        $class_id = $request->student_class;
        $section = $request->section;
        if (empty($class_id) && empty($section) && empty($student_id_no)) {
            return collect();
        }
        
        $academicYear = getDefaultAcademicYearValue();
        $fromDate = $academicYear . '-01-01';
        $toDate = $academicYear . '-12-31';

        $branch_id = auth()->user()->branch_id;
    
        $classFeesQuery = DB::table('class_fees')
            ->join('fee_types as ft', 'class_fees.fee_type_id', '=', 'ft.id')
            ->join('student_classes as sc', 'class_fees.student_class_id', '=', 'sc.id')
            ->leftJoin('invoice_items as ii', function ($join) use ($fromDate, $toDate) {
                $join->on('si.id', '=', 'ii.student_id')
                    ->whereColumn('class_fees.fee_type_id', 'ii.fee_type_id')
                    ->whereBetween('ii.payment_date', [$fromDate, $toDate]);
            })
            ->whereNotIn('ft.code', ['Hostel', 'Transport'])
            ->when($student_id_no, function ($q) use ($student_id_no) {
                $q->where('si.student_id_no', '=', $student_id_no);
            })
            ->when($class_id, function ($q) use ($class_id) {
                $q->where('class_fees.student_class_id', '=', $class_id);
            })
            ->when($section, function ($q) use ($section) {
                $q->where('si.section_id', '=', $section);
            })
            ->where('class_fees.branch_id', $branch_id)
            ->groupBy('ft.id', 'ft.title', 'ii.month', 'ft.code', 'class_fees.amount', 'si.id', 'si.first_name', 'si.last_name', 'si.name', 'si.student_id_no', 'sc.class_name')
            ->select([
                'si.id',
                'si.first_name',
                'si.last_name',
                'si.name',
                'si.student_id_no',
                'sc.class_name',
                'ft.id as fee_type_id',
                'ft.title',
                'ii.month',
                'ii.payment_date',
                'ft.code',
                'class_fees.amount as assigned_fee',
                DB::raw('COALESCE(SUM(ii.amount), 0) as paid_fee'),
                DB::raw('class_fees.amount - COALESCE(SUM(ii.amount), 0) as due_fee'),
            ]);
            
        $studentFeesQuery = DB::table('student_fees')
            ->join('student_infos as si', 'si.id', '=', 'student_fees.student_id')
            ->leftJoin('invoice_items as ii', function ($join) use ($fromDate, $toDate) {
                $join->on('student_fees.student_id', '=', 'ii.student_id')
                    ->whereColumn('student_fees.id', 'ii.student_fee_id')
                    ->whereBetween('ii.payment_date', [$fromDate, $toDate]);
            })
            ->join('fee_types as ft', 'ii.fee_type_id', '=', 'ft.id')
            ->join('student_classes as sc', 'si.student_class_id', '=', 'sc.id')
            ->whereNotIn('ft.code', ['Hostel', 'Transport'])
            ->when($student_id_no, function ($q) use ($student_id_no) {
                $q->where('si.student_id_no', '=', $student_id_no);
            })
            ->when($class_id, function ($q) use ($class_id) {
                $q->where('si.student_class_id', '=', $class_id);
            })
            ->when($section, function ($q) use ($section) {
                $q->where('si.section_id', '=', $section);
            })
            ->groupBy('ft.id', 'ft.title', 'ii.month', 'ft.code', 'student_fees.total_amount', 'si.id', 'si.first_name', 'si.last_name', 'si.name', 'si.student_id_no', 'sc.class_name')
            ->select([
                'si.id',
                'si.first_name',
                'si.last_name',
                'si.name',
                'si.student_id_no',
                'sc.class_name',
                'ft.id as fee_type_id',
                'ft.title',
                'ii.month',
                'ii.payment_date',
                'ft.code',
                'student_fees.total_amount as assigned_fee',
                DB::raw('COALESCE(SUM(ii.amount), 0) as paid_fee'),
                DB::raw('student_fees.total_amount - COALESCE(SUM(ii.amount), 0) as due_fee'),
            ]);
    
        $paymentDetails = $classFeesQuery->unionAll($studentFeesQuery)->get();
        return $paymentDetails;
    }
    
    

    public function paymentDetailsWithColumns(Request $request)
    {
        $data = $this->paymentDetailsReport($request);
        return $data->map(function ($student) {
            return [
                'name' => $student->name,
                'student_id_no' => $student->student_id_no,
                'class_name' => $student->class_name,
                'section_name' => $student->section_name ?? 'N/A',
                'fee_title' => $student->title,
                'payment_date' => $student->payment_date ?? 'N/A',
                'total_assigned_fee' => $student->assigned_fee,
                'total_paid_fee' => $student->paid_fee,
                'total_due_fee' => $student->due_fee,
            ];
        });
    }

    public function getDueFeesReport(Request $request)
    {
        // Check if the request has any filters
        if (empty($request->all())) {
            return collect();
        }
        $student_id_no = $request->query('student_id_no');
        $student_class = $request->query('student_class');
        $studentInfo = [];
        if($student_id_no){
            $studentInfo = StudentInfo::where('student_id_no', $student_id_no)->first();
        }
        $section = $request->query('section');
        $branch_id = auth()->user()->branch_id;
        $academicYear = getDefaultAcademicYearID();
        $dateRange = $request->date;
        list($fromDate, $toDate) = parseDateRange($dateRange);

        //all months array from date range
        $months = getMonthsFromDateRange($fromDate, $toDate);

        $classFeesQuery = DB::table('class_fees')
            ->join('fee_types as ft', 'class_fees.fee_type_id', '=', 'ft.id')
            ->join('student_classes as sc', 'class_fees.student_class_id', '=', 'sc.id')
            ->join('student_infos as si', 'sc.id', '=', 'si.student_class_id')
            ->join('sections as sec', 'si.section_id', '=', 'sec.id')
            ->leftJoin('invoice_items as ii', function ($join) use ($fromDate, $toDate) {
                $join->on('si.id', '=', 'ii.student_id')
                    ->whereColumn('class_fees.fee_type_id', 'ii.fee_type_id')
                    ->whereBetween('ii.payment_date', [$fromDate, $toDate]);
            })
            ->whereNotIn('ft.code', ['Hostel', 'Transport'])
            ->when($studentInfo, function ($q) use ($studentInfo) {
                $q->where('si.student_id_no', '=', $studentInfo->student_id_no);
            })
            ->when($section, function ($q) use ($section) {
                $q->where('si.section_id', '=', $section);
            })
            ->where('si.branch_id', $branch_id)
            ->where('class_fees.branch_id', $branch_id)
            ->where('class_fees.academic_year_id', $academicYear)
            ->whereIn('class_fees.month', $months)
            ->when($student_class, function ($q) use ($student_class) {
                $q->where('class_fees.student_class_id', '=', $student_class);
            })
            ->groupBy('si.id', 'si.name', 'si.student_id_no', 'sc.class_name')
            ->select([
                'si.id',
                'si.name',
                'si.student_id_no',
                'sc.class_name',
                'sec.section_name',
                DB::raw('SUM(class_fees.amount) as total_assigned_fee'),
                DB::raw('COALESCE(SUM(ii.amount), 0) as total_paid_fee'),
                DB::raw('SUM(class_fees.amount) - COALESCE(SUM(ii.amount), 0) as total_due_fee')
            ]);

        // Student-specific fees query (hostel and transport)
        $studentFeesQuery = DB::table('student_fees')
            ->join('student_infos as si', 'si.id', '=', 'student_fees.student_id')
            ->leftJoin('invoice_items as ii', function ($join) use ($fromDate, $toDate) {
                $join->on('student_fees.student_id', '=', 'ii.student_id')
                    ->whereColumn('student_fees.month', 'ii.month')
                    ->whereBetween('ii.payment_date', [$fromDate, $toDate]);
            })
            ->join('fee_types as ft', 'ii.fee_type_id', '=', 'ft.id')
            ->join('student_classes as sc', 'si.student_class_id', '=', 'sc.id')

            ->join('sections as sec', 'si.section_id', '=', 'sec.id')
            ->whereNotIn('ft.code', ['Hostel', 'Transport'])
            ->when($studentInfo, function ($q) use ($studentInfo) {
                $q->where('student_fees.student_id', '=', $studentInfo->id);
            })
            ->when($student_class, function ($q) use ($student_class) {
                $q->where('si.student_class_id', '=', $student_class);
            })
            ->when($section, function ($q) use ($section) {
                $q->where('si.section_id', '=', $section);
            })
            ->groupBy('si.id', 'si.name', 'si.student_id_no', 'sc.class_name')
            ->select([
                'si.id',
                'si.name',
                'si.student_id_no',
                'sc.class_name',
                'sec.section_name',
                DB::raw('SUM(student_fees.total_amount) as total_assigned_fee'),
                DB::raw('COALESCE(SUM(ii.amount), 0) as total_paid_fee'),
                DB::raw('SUM(student_fees.total_amount) - COALESCE(SUM(ii.amount), 0) as total_due_fee')
            ]);
        // Combine both queries using UNION ALL
        $finalQuery = $classFeesQuery->unionAll($studentFeesQuery);

        // Get the total summary
        $feeSummary = DB::query()
            ->fromSub($finalQuery, 'combined_fees')
            ->groupBy('id', 'name', 'student_id_no', 'class_name')
            ->select([
                'id',
                'name',
                'student_id_no', 
                'class_name',
                'section_name',
                DB::raw('SUM(total_assigned_fee) as total_assigned_fee'),
                DB::raw('SUM(total_paid_fee) as total_paid_fee'),
                DB::raw('SUM(total_due_fee) as total_due_fee')
            ])
            ->get();
        return $feeSummary;
    }

    public function getDueFeesReportWithColumns(Request $request)
    {
        $data = $this->getDueFeesReport($request);
        return $data->map(function ($student) {
            return [
                'name' => $student->name,
                'student_id_no' => $student->student_id_no,
                'class_name' => $student->class_name,
                'section_name' => $student->section_name ?? 'N/A',
                'total_assigned_fee' => $student->total_assigned_fee,
                'total_paid_fee' => $student->total_paid_fee,
                'total_due_fee' => $student->total_due_fee,
            ];
        });
    }

    public function getIncomeReport(Request $request)
    {
        $dateRange = $request->date_range;
        list($fromDate, $toDate) = parseDateRange($dateRange);

        // Group by income_date and order by income_date
        $incomes = Income::selectRaw('income_date, incomes.accounting_head_id, accounting_heads.name as accounting_head_name, SUM(incomes.amount) as total_amount')
            ->join('accounting_heads', 'incomes.accounting_head_id', '=', 'accounting_heads.id')
            ->whereBetween('income_date', [$fromDate, $toDate])
            ->groupBy('income_date', 'incomes.accounting_head_id', 'accounting_heads.name')
            ->orderBy('income_date')
            ->orderBy('incomes.accounting_head_id')
            ->get();
        return $incomes;
    }

    public function getIncomeReportWithColumns(Request $request)
    {
        $data = $this->getIncomeReport($request);
        return $data->map(function ($income) {
            return [
                'income_date' => $income->income_date,
                'accounting_head_name' => $income->accounting_head_name,
                'total_amount' => $income->total_amount,
            ];
        });
    }

    public function getExpenseReport(Request $request)
    {
        $dateRange = $request->date_range;
        list($fromDate, $toDate) = parseDateRange($dateRange);

        // Group by expense_date and order by expense_date
        $expenses = Expense::selectRaw('expense_date, expenses.accounting_head_id, accounting_heads.name as accounting_head_name, SUM(expenses.amount) as total_amount')
            ->join('accounting_heads', 'expenses.accounting_head_id', '=', 'accounting_heads.id')
            ->whereBetween('expense_date', [$fromDate, $toDate])
            ->groupBy('expense_date', 'expenses.accounting_head_id', 'accounting_heads.name')
            ->orderBy('expense_date')
            ->orderBy('expenses.accounting_head_id')
            ->get();
        return $expenses;
    }

    public function getExpenseReportWithColumns(Request $request)
    {
        $data = $this->getExpenseReport($request);
        return $data->map(function ($expense) {
            return [
                'expense_date' => $expense->expense_date,
                'accounting_head_name' => $expense->accounting_head_name,
                'total_amount' => $expense->total_amount,
            ];
        });
    }

    public function getDateWiseIncome(Request $request)
    {
        $dateRange = $request->date_range;
        list($fromDate, $toDate) = parseDateRange($dateRange);
    
        // Group by income_date and calculate the total income for each day
        $incomes = Income::selectRaw('income_date, SUM(incomes.amount) as total_amount')
            ->whereBetween('income_date', [$fromDate, $toDate])
            ->groupBy('income_date')
            ->orderBy('income_date')
            ->get();
    
        // Calculate the total sum of income
        $totalIncome = Income::whereBetween('income_date', [$fromDate, $toDate])
            ->sum('amount');
    
        return [
            'incomes' => $incomes,
            'totalIncome' => $totalIncome
        ];
    }
    

    public function getDateWiseExpense(Request $request)
    {
        $dateRange = $request->date_range;
        list($fromDate, $toDate) = parseDateRange($dateRange);
    
        // Group by expense_date and calculate the total expense for each day
        $expenses = Expense::selectRaw('expense_date, SUM(expenses.amount) as total_amount')
            ->whereBetween('expense_date', [$fromDate, $toDate])
            ->groupBy('expense_date')
            ->orderBy('expense_date')
            ->get();
    
        // Calculate the total sum of expenses
        $totalExpense = Expense::whereBetween('expense_date', [$fromDate, $toDate])
            ->sum('amount');
    
        return [
            'expenses' => $expenses,
            'totalExpense' => $totalExpense
        ];
    }

    public function getIncomeHeadwiseReport(Request $request)
    {
        $dateRange = $request->date_range;
        list($fromDate, $toDate) = parseDateRange($dateRange);
        
        $incomes = Income::selectRaw('income_date, incomes.accounting_head_id, accounting_heads.name as accounting_head_name, SUM(incomes.amount) as total_amount')
            ->join('accounting_heads', 'incomes.accounting_head_id', '=', 'accounting_heads.id')
            ->whereBetween('income_date', [$fromDate, $toDate])
            ->groupBy('income_date', 'incomes.accounting_head_id', 'accounting_heads.name')
            ->orderBy('income_date')
            ->orderBy('incomes.accounting_head_id')
            ->get();
        return $incomes;
    }

    public function getExpenseHeadwiseReport(Request $request)
    {
        $dateRange = $request->date_range;
        list($fromDate, $toDate) = parseDateRange($dateRange);
        
        $expenses = Expense::selectRaw('expense_date, expenses.accounting_head_id, accounting_heads.name as accounting_head_name, SUM(expenses.amount) as total_amount')
            ->join('accounting_heads', 'expenses.accounting_head_id', '=', 'accounting_heads.id')
            ->whereBetween('expense_date', [$fromDate, $toDate])
            ->groupBy('expense_date', 'expenses.accounting_head_id', 'accounting_heads.name')
            ->orderBy('expense_date')
            ->orderBy('expenses.accounting_head_id')
            ->get();
        return $expenses;
    }

    public function getSalaryReport(Request $request)
    {
        if (empty($request->all())) {
            return [collect(), collect()];
        }

        $dateRange = $request->date_range;
        list($fromDate, $toDate) = parseDateRange($dateRange);

        $newSalaries = MonthlySalary::with('employee', 'employee.branch', 'employee.shift', 'salaryGrade.salaryGradeRules')
            // ->where('month', $request->month)
            // ->where('academic_year_id', $request->academicYear)
            ->whereBetween('process_date', [$fromDate, $toDate])
            ->get();

        return [$newSalaries, SalaryHead::get()];
    }

    public function getRelatedData()
    {
        return [
            'studentClasses' => StudentClass::pluck('class_name', 'id'),
            'groups' => Group::pluck('group_name', 'id'),
            'shifts' => Shift::pluck('shift_name', 'id'),
            'sections' => Section::pluck('section_name', 'id'),
            'academicYears' => AcademicYear::pluck('year', 'id'),
        ];
    }
}
