<?php

namespace App\Services\Reports;

use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Models\Academic\Branch;
use Illuminate\Support\Facades\DB;


class MultiBranchReportService
{

    public function getTodaysTotalCollection()
    {
        $from_date = date('Y-m-01');
        $to_date = date('Y-m-t');

        $query = DB::table('invoices')
            ->select(
                DB::raw('SUM(invoices.paid_amount) as total_amount')
            )
            ->whereBetween(DB::raw('DATE(invoices.created_at)'), [$from_date, $to_date]);

        return $query->first()->total_amount;

    }

    public function getDailyCollectionReport(Request $request)
    {
        $from_date = $request->from_date ?? date('Y-01-01');
        $to_date = $request->to_date ?? date('Y-12-31');

        $query = DB::table('invoices')
            ->select(
                'branches.branch_name as branch_name',
                DB::raw('SUM(invoices.paid_amount) as total_amount')
            )
            ->join('branches', 'branches.id', '=', 'invoices.branch_id')
            ->whereBetween(DB::raw('DATE(invoices.invoice_date)'), [$from_date, $to_date])
            ->groupBy('branches.id');

        return [
            'collections' => $query->get(),
            'branches' => Branch::all()
        ];
    }

    public function getTotalCollection()
    {
        $from_date = Carbon::now()->subDays(29)->toDateString(); // 30 days ago
        $to_date = Carbon::now()->toDateString(); // today

        $query = DB::table('invoices')
            ->select(
                DB::raw('SUM(invoices.paid_amount) as total_amount')
            )
            ->whereBetween(DB::raw('DATE(invoices.created_at)'), [$from_date, $to_date]);

        return $query->first()->total_amount;
    }

    public function getTotalCollectionReport(Request $request)
    {

        list($fromDate, $toDate) = parseDateRange($request->date);
        $from_date = $fromDate ?? date('Y-01-01');
        $to_date = $toDate ?? date('Y-12-31');
        $query = DB::table('branches')
                ->select(
                    'branches.branch_name as branch_name',
                    DB::raw('COALESCE(SUM(invoices.paid_amount), 0) as total_amount')
                )
                ->leftJoin('invoices', function ($join) use ($from_date, $to_date) {
                    $join->on('invoices.branch_id', '=', 'branches.id')
                        ->whereBetween(DB::raw('DATE(invoices.invoice_date)'), [$from_date, $to_date]);
                })
                ->groupBy('branches.id');

        return [
            'collections' => $query->get(),
            'branches' => Branch::all()
        ];
    }


    public function getTodaysTotalIncome()
    {
        $from_date = date('Y-m-d');
        $to_date = date('Y-m-d');

        $query = DB::table('incomes')
            ->select(
                DB::raw('SUM(incomes.amount) as total_amount')
            )
            ->whereBetween(DB::raw('DATE(incomes.income_date)'), [$from_date, $to_date]);

        return $query->first()->total_amount;
    }

    public function getTodaysTotalExpense()
    {
        $from_date = date('Y-m-d');
        $to_date = date('Y-m-d');

        $from_date = Carbon::now()->subDays(29)->toDateString(); // 30 days ago
        $to_date = Carbon::now()->toDateString(); // today
  
        $query = DB::table('expenses')
            ->select(
                DB::raw('SUM(expenses.amount) as total_amount')
            )
            ->whereBetween(DB::raw('DATE(expenses.expense_date)'), [$from_date, $to_date]);

        return $query->first()->total_amount;
    }

    public function getTotalIncome()
    {
        $from_date = Carbon::now()->subDays(29)->toDateString(); // 30 days ago
        $to_date = Carbon::now()->toDateString(); // today

        $query = DB::table('incomes')
            ->select(
                DB::raw('SUM(incomes.amount) as total_amount')
            )
            ->whereBetween(DB::raw('DATE(incomes.income_date)'), [$from_date, $to_date]);

        return $query->first()->total_amount;
    }

    public function getTotalExpense()
    {
        $from_date = Carbon::now()->subDays(29)->toDateString(); // 30 days ago
        $to_date = Carbon::now()->toDateString(); // today

        $query = DB::table('expenses')
            ->select(
                DB::raw('SUM(expenses.amount) as total_amount')
            )
            ->whereBetween(DB::raw('DATE(expenses.expense_date)'), [$from_date, $to_date]);

        return $query->first()->total_amount;
    }


    public function getIncomeReport(Request $request)
    {
        list($fromDate, $toDate) = parseDateRange($request->date);
        $from_date = $fromDate ?? Carbon::now()->subDays(29)->toDateString();
        $to_date = $toDate ?? Carbon::now()->toDateString();

        $query = DB::table('branches')
        ->select(
            'branches.branch_name as branch_name',
            DB::raw('COALESCE(SUM(incomes.amount), 0) as total_amount')
        )
        ->leftJoin('incomes', function ($join) use ($from_date, $to_date) {
            $join->on('incomes.branch_id', '=', 'branches.id')
                ->whereBetween('incomes.income_date', [$from_date, $to_date]);
        })
        ->groupBy('branches.id');
    

        return [
            'incomes' => $query->get(),
            'branches' => Branch::all()
        ];
    }

    public function getExpenseReport(Request $request)
    {
        list($fromDate, $toDate) = parseDateRange($request->date);
        $from_date = $fromDate ?? Carbon::now()->subDays(29)->toDateString();
        $to_date = $toDate ?? Carbon::now()->toDateString();

        $query = DB::table('branches')
            ->select(
                'branches.branch_name as branch_name',
                DB::raw('COALESCE(SUM(expenses.amount), 0) as total_amount')
            )
            ->leftJoin('expenses', function ($join) use ($from_date, $to_date) {
                $join->on('expenses.branch_id', '=', 'branches.id')
                    ->whereBetween('expenses.expense_date', [$from_date, $to_date]);
            })
            ->groupBy('branches.id');

        return [
            'expenses' => $query->get(),
            'branches' => Branch::all()
        ];
    }

    public function getBranchIncomeSummary()
    {
        $from_date = Carbon::now()->subDays(29)->toDateString(); // 30 days ago
        $to_date = Carbon::now()->toDateString(); // today

        $query = DB::table('branches')
        ->select(
            'branches.branch_name as branch_name',
            DB::raw('COALESCE(SUM(incomes.amount), 0) as total_amount')
        )
        ->leftJoin('incomes', function ($join) use ($from_date, $to_date) {
            $join->on('incomes.branch_id', '=', 'branches.id')
                ->whereBetween('incomes.income_date', [$from_date, $to_date]);
        })
        ->groupBy('branches.id');

        return $query->get();
    }

    public function getBranchExpenseSummary()
    {
        $from_date = Carbon::now()->subDays(29)->toDateString(); // 30 days ago
        $to_date = Carbon::now()->toDateString(); // today

        $query = DB::table('branches')
        ->select(
            'branches.branch_name as branch_name',
            DB::raw('COALESCE(SUM(expenses.amount), 0) as total_amount')
        )
        ->leftJoin('expenses', function ($join) use ($from_date, $to_date) {
            $join->on('expenses.branch_id', '=', 'branches.id')
                ->whereBetween('expenses.expense_date', [$from_date, $to_date]);
        })
        ->groupBy('branches.id');

        return $query->get();
    }

    
} 
