<?php
namespace App\Services\Reports;

use App\Filters\StudentReportFilter;
use App\Models\Academic\AcademicYear;
use App\Models\Academic\ClassSection;
use App\Models\Academic\Group;
use App\Models\Academic\Section;
use App\Models\Academic\Shift;
use App\Models\Academic\StudentClass;
use App\Models\Student\StudentInfo;

class StudentReportService
{

    public function getStudentReport(array $queryParams = [])
    {
        if(empty($queryParams)){
            return [];
        }
        $queryBuilder = StudentInfo::with(['branch', 'shift', 'section', 'studentClass', 'studentHostel', 'studentTransport'])->latest();

        $students = resolve(StudentReportFilter::class)->getResults([
            'builder' => $queryBuilder,
            'params' => $queryParams
        ]);
        return $students->get();
    }

    public function getStudentReportWithColumns(array $queryParams = [])
    {
        $students = $this->getStudentReport($queryParams);
        return $students->map(function ($student) {
            return [
                'student_id_no' => $student->student_id_no,
                'profile_image' => $student->profile_image ?? 'uploads/student/graduated_student.png',
                'name' => $student->name,
                'class_roll' => $student->class_roll,
                'student_class.class_name' => $student->studentClass->class_name,
                'section.section_name' => $student->section?->section_name ?? '--',
                'shift.shift_name' => $student->shift?->shift_name ?? '--',
                'branch.branch_name' => $student->branch?->branch_name ?? '--',
                'father_name' => $student->father_name ?? '--',
                'father_mobile' => $student->father_mobile ?? '--',
                'mother_name' => $student->mother_name ?? '--',
                'mother_mobile' => $student->mother_mobile ?? '--',
                'guardian_name' => $student->guardian_name ?? '--',
                'guardian_mobile' => $student->guardian_mobile ?? '--',
                'mobile_no' => $student->mobile_no ?? '--',
            ];
        });
    }

    public function getClassSectionReport()
    {
        $totalStudentsBySection = ClassSection::withCount('students')
        ->with(['studentClass', 'section', 'shift'])->get();
        return $totalStudentsBySection;
    }

    public function getClassSectionReportWithColumns()
    {
        $totalStudentsBySection = $this->getClassSectionReport();
        $totalStudents = 0;
        $totalStudentsBySection = $totalStudentsBySection->map(function ($section, $index) use (&$totalStudents) {
            $totalStudents += $section->students_count ?? '0';
            return [
                'sn' => $index + 1,
                'class_name' => $section->studentClass->class_name,
                'section_name' => $section->section?->section_name ?? '--',
                'shift_name' => $section?->shift?->shift_name ?? '--',
                'total_students' => $section->students_count ?? '0',
            ];
        });
        $totalStudentsBySection->grand_total = $totalStudents;
        return $totalStudentsBySection;
    }

    public function getSiblingReport(array $queryParams = [])
    {
        if(empty($queryParams)){
            return [];
        }
        $queryBuilder = StudentInfo::with(['branch', 'shift', 'section', 'studentClass', 'siblings'])
        ->orderByRaw('class_roll IS NULL, class_roll ASC');

        $students = resolve(StudentReportFilter::class)->getResults([
            'builder' => $queryBuilder,
            'params' => $queryParams
        ])->get();

        foreach ($students as $student) {
            $student->all_siblings = $student->allSiblings();
        }
        
        return $students;
    }

    public function getSiblingReportWithColumns(array $queryParams = [])
    {
        $students = $this->getSiblingReport($queryParams);
        return $students->map(function ($student) {
            return [
                'student_id_no' => $student->student_id_no,
                'name' => $student->name,
                'class_roll' => $student->class_roll,
                'mobile_no' => $student->mobile_no,
                'father_name' => $student->father_name,
                'mother_name' => $student->mother_name,
                //make a comma separated string of all siblings
                'all_siblings' => $student->all_siblings->pluck('name')->implode(', '),
            ];
        });
    }


    public function getRelatedData()
    {
        return [
            'studentClasses' => StudentClass::pluck('class_name', 'id'),
            'groups' => Group::pluck('group_name', 'id'),
            'shifts' => Shift::pluck('shift_name', 'id'),
            'sections' => Section::pluck('section_name', 'id'),
            'academicYears' => AcademicYear::pluck('year', 'id'),
        ];
    }

}