<?php
namespace App\Services\Student;
use App\Constants\Status;
use App\Models\Result\Exam;
use App\Models\Result\Grade;
use Illuminate\Http\Request;
use App\Models\HrPayroll\Leave;
use App\Models\Result\MarkHead;
use App\Models\Academic\Section;
use App\Models\Result\ExamRecord;
use App\Models\Result\ExamPublish;
use Illuminate\Support\Facades\DB;
use App\Models\HrPayroll\LeaveType;
use App\Models\Student\StudentInfo;
use App\Models\Academic\AcademicYear;
use App\Models\Academic\ClassRoutine;
use App\Models\Academic\StudentClass;
use App\Models\Result\MarkHeadAssign;
use App\Models\Academic\SubjectsAssign;
use App\Models\Scopes\BranchFilterScope;
use App\Models\Student\StudentPromotion;
use App\Models\Attendance\StudentAttendance;
class StudentDashboardService
{
    public function getStudentInfo()
    {
        $user_id = auth()->user()->id;
        $studentInfo = StudentInfo::withoutGlobalScope(BranchFilterScope::class)->with([
            'shift',
            'studentClass.subjectAssignments',
            'section',
            'group',
        ])->where('user_id', $user_id)->first();
        return $studentInfo;
    }
    public function getClassRoutine()
    {
        $studentInfo = $this->getStudentInfo();
        $classRoutine = ClassRoutine::with(['routineDetails', 'routineDetails.teacher', 'routineDetails.subject'])->where('student_class_id', $studentInfo->student_class_id)->first();
        return $classRoutine;
    }
    public function getMySubjects()
    {
        $studentInfo = $this->getStudentInfo();
        $mySubjects = $studentInfo->studentClass->subjectAssignments->load('subject');
        return $mySubjects;
    }
    public function getMyFriends()
    {
        $studentInfo = $this->getStudentInfo();
        $myFriends = StudentInfo::withoutGlobalScope(BranchFilterScope::class)->where('student_class_id', $studentInfo->student_class_id)
            ->where('id', '!=', $studentInfo->id)
            ->where('section_id', $studentInfo->section_id)
            ->when(!empty($studentInfo->group_id), function ($query) use ($studentInfo) {
                $query->where('group_id', $studentInfo->group_id);
            })
            ->when(!empty($studentInfo->shift_id), function ($query) use ($studentInfo) {
                $query->where('shift_id', $studentInfo->shift_id);
            })
            ->orderBy('class_roll', 'asc')
            ->get();
        return $myFriends;
    }
    public function getMyApprovedLeaves()
    {
        $studentInfo = $this->getStudentInfo();
        $myApprovedLeaves = Leave::where('user_id', $studentInfo->user_id)->where('status', Status::APPROVED)->get();
        return $myApprovedLeaves;
    }
    public function getMyCurrentAttendance()
    {
        $studentInfo = $this->getStudentInfo();
        $myCurrentAttendance = StudentAttendance::where('student_id', $studentInfo->id)
            ->where('is_present', 1)
            ->where('punch_date', '>=', now()->startOfDay())
            ->where('punch_date', '<=', now()->endOfDay())
            ->get();
        return $myCurrentAttendance;
    }
    public function getMyAttendanceHistory()
    {
        $studentInfo = $this->getStudentInfo();
        $myAttendanceHistory = StudentAttendance::where('student_id', $studentInfo->id)->get();
        return $myAttendanceHistory;
    }
    public function getMyMarkSheet(Request $request)
    {
        $studentInfo = $this->getStudentInfo();
        // Get exam records for the student
        $exam_id = $request->exam_id ?? getCurrentActiveExamId();
        $academic_year = Exam::find($exam_id)->year ?? getDefaultAcademicYearID();
        $studentPromotion = StudentPromotion::where('student_id', $studentInfo->id)
            ->where('from_academic_year_id', $academic_year)->first();
        $student_class_id = $studentPromotion->from_class_id ?? $studentInfo->student_class_id;
        // Get assigned marks head for the class
        $assignedMarksHead = MarkHeadAssign::where('exam_id', $exam_id)
            ->where('student_class_id', $student_class_id)
            ->pluck('head_marks', 'subject_id');
        // Fetch only the current student with their exam records
        $student = $studentInfo->where('id',$studentInfo->id)->whereHas('examRecords', function ($query) use ($exam_id, $academic_year) {
                $query->where('exam_id', $exam_id)
                    ->where('year', $academic_year);
            })
            ->with(['examRecords.subject', 'examRecords' => function ($query) use ($exam_id, $academic_year) {
                $query->where('exam_id', $exam_id)
                    ->where('year', $academic_year)
                    ->orderBy('subject_id');
            }])
            ->first();
        // dd($request->all(), $studentInfo, $student, $academic_year);
        // Query for the highest marks in the class (keeping this for comparison)
        $highestMarks = ExamRecord::where('class_id', $student_class_id)
            ->where('exam_id', $exam_id)
            ->where('shift_id', $studentInfo->shift_id)
            ->where('section_id', $studentInfo->section_id)
            ->where('group_id', $studentInfo->group_id)
            ->groupBy('subject_id')
            ->select('subject_id', DB::raw('MAX(total_marks) as highest_marks'))
            ->get();
        $highest_marks_array = [];
        foreach ($highestMarks as $marks) {
            $highest_marks_array[$marks->subject_id] = $marks->highest_marks;
        }
        $grades = Grade::orderBy('mark_from', 'desc')->get();
        $markHeads = MarkHead::all();
        // $exams = Exam::pluck('exam_name', 'id');
        $exams = Exam::orderBy('id', 'desc')->select('id', 'exam_name', 'year')->get();
        $academicYears = AcademicYear::pluck('year', 'id');
        $publish_date = ExamPublish::where('exam_id', $exam_id)->first()?->publish_date;
        $studentClasses = StudentClass::pluck('class_name', 'id');
        $sections = Section::pluck('section_name', 'id');
        $subjects_assigns = SubjectsAssign::where('student_class_id', $student_class_id)->select('id','student_class_id', 'subject_id')->get();
        $selectedExam = Exam::find($exam_id);
        // dd($student, $studentInfo,$exams, $exam_id);
        return [
            'student' => $student,
            'markHeads' => $markHeads,
            'grades' => $grades,
            'assignedMarksHead' => $assignedMarksHead,
            'highestMarks' => $highestMarks,
            'exams' => $exams,
            'selectedExam' => $selectedExam,
            'academicYears' => $academicYears,
            'exam_id' => $exam_id,
            'academic_year' => $academic_year,
            'highest_marks_array' => $highest_marks_array,
            'publish_date' => $publish_date,
            'studentClasses' => $studentClasses,
            'sections' => $sections,
            'studentPromotion' => $studentPromotion,
            'subjects_assigns' => $subjects_assigns,
        ];
    }
    public function getOthersMarkSheet(Request $request)
    {
        //here we will get the mark sheet of other students
        $studentInfo = $this->getStudentInfo();
        // Get exam records for the student
         $exam_id = $request->exam_id ?? getCurrentActiveExamId();
        $academic_year = Exam::find($exam_id)->year ?? getDefaultAcademicYearID();
        $studentPromotion = StudentPromotion::where('student_id', $studentInfo->id)
            ->where('from_academic_year_id', $academic_year)->first();
        $student_class_id = $studentPromotion->from_class_id ?? $studentInfo->student_class_id;
        $section_id = $studentPromotion->from_section_id ?? $studentInfo->section_id;
        $shift_id = $studentPromotion->from_shift_id ?? '';
        $group_id = $studentPromotion->from_group_id ?? $studentInfo->group_id;
        $assignedMarksHead = MarkHeadAssign::where('exam_id', $exam_id)->where('student_class_id', $student_class_id)->pluck('head_marks', 'subject_id');
        $examRecords = ExamRecord::where('exam_id', $exam_id)
            ->where('year', $academic_year)
            ->where('class_id', $student_class_id)
            ->when(!empty($shift_id), function ($query) use ($shift_id) {
                $query->where('shift_id', $shift_id);
            })
            ->when(!empty($section_id), function ($query) use ($section_id) {
                $query->where('section_id', $section_id);
            })
            ->when(!empty($group_id), function ($query) use ($group_id) {
                $query->where('group_id', $group_id);
            });
        // Get unique student IDs from exam records
        $studentIds = $examRecords->pluck('student_id')->unique();
        // Get students with their exam records
        $students = StudentInfo::withoutGlobalScope(BranchFilterScope::class)
            ->whereIn('id', $studentIds)
            ->with(['examRecords' => function ($query) use ($exam_id, $academic_year, $student_class_id) {
                $query->where('exam_id', $exam_id)
                    ->where('year', $academic_year)
                    ->where('class_id', $student_class_id)
                    ->orderBy('subject_id');
            }, 'examRecords.subject'])
            ->orderBy('class_roll', 'ASC')
            ->get();
        // Calculate highest marks
        $highestMarks = ExamRecord::where('class_id', $student_class_id)
            ->where('exam_id', $exam_id)
            ->where('year', $academic_year)
            ->when(!empty($shift_id), function ($query) use ($shift_id) {
                $query->where('shift_id', $shift_id);
            })
            ->when(!empty($section_id), function ($query) use ($section_id) {
                $query->where('section_id', $section_id);
            })
            ->when(!empty($group_id), function ($query) use ($group_id) {
                $query->where('group_id', $group_id);
            })
            ->groupBy('subject_id')
            ->select('subject_id', DB::raw('MAX(total_marks) as highest_marks'))
            ->get();
        $highest_marks_array = $highestMarks->pluck('highest_marks', 'subject_id')->toArray();
        $grades = Grade::orderBy('mark_from', 'desc')->get();
        $markHeads = MarkHead::all();
        $exams = Exam::orderBy('id', 'desc')->select('id', 'exam_name', 'year')->get();
        $academicYears = AcademicYear::pluck('year', 'id');
        $publish_date = Exam::where('id', $exam_id)->first()?->publish_date;

        $subjects_assigns = SubjectsAssign::where('student_class_id', $student_class_id)->select('id','student_class_id', 'subject_id')->get();
        $selectedExam = Exam::find($exam_id);
        // dd($highest_marks_array);
        return [
            'students' => $students,
            'markHeads' => $markHeads,
            'grades' => $grades,
            'assignedMarksHead' => $assignedMarksHead,
            'highestMarks' => $highestMarks,
            'exams' => $exams,
            'subjects_assigns' => $subjects_assigns,
            'selectedExam' => $selectedExam,
            'exam_id' => $exam_id,
            'academic_year' => $academic_year,
            'highest_marks_array' => $highest_marks_array,
            'publish_date' => $publish_date,
            'studentInfo' => $studentInfo,
            'studentPromotion' => $studentPromotion,
            'academicYears' => $academicYears,
        ];
    }
    public function getFeesSummary()
    {
        $studentInfo = $this->getStudentInfo();
        $academicYearID = getDefaultAcademicYearID();
        $monthlyFees = collect();
        $months = range(1, date('n'));
        $grandTotal = 0;
        $grandPaid  = 0;
        foreach ($months as $month) {
            // Assigned fees
            $total = DB::table('class_fees')
                ->where('branch_id', $studentInfo->branch_id)
                ->where('academic_year_id', $academicYearID)
                ->where('student_class_id', $studentInfo->student_class_id)
                ->where('month', $month)
                ->sum('amount');
            // Paid fees
            $paid = DB::table('student_fees')
                ->where('branch_id', $studentInfo->branch_id)
                ->where('academic_year_id', $academicYearID)
                ->where('student_id', $studentInfo->id)
                ->where('month', $month)
                ->sum('total_amount');
            // Skip empty months
            if ($total == 0 && $paid == 0) {
                continue;
            }
            $grandTotal += $total;
            $grandPaid  += $paid;
            $monthlyFees->push([
                'month'        => $month,
                'total_amount' => number_format($total, 2, '.', ''),
                'paid_amount'  => number_format($paid, 2, '.', ''),
                'due_amount'   => number_format($total - $paid, 2, '.', ''),
            ]);
        }
        // dd($monthlyFees);
        return (object)[
            'student' => [
                'id'             => $studentInfo->id,
                'name'           => $studentInfo->name,
                'student_id_no'  => $studentInfo->student_id_no,
                'class_name'     => optional($studentInfo->studentClass)->class_name,
            ],
            'monthly_fees' => $monthlyFees,
            'grand_total_amount' => number_format($grandTotal, 2, '.', ''),
            'grand_paid_amount'  => number_format($grandPaid, 2, '.', ''),
            'grand_due_amount'   => number_format($grandTotal - $grandPaid, 2, '.', ''),
        ];
    }
    public function getMyFees()
    {
        $studentInfo = $this->getStudentInfo();
        $academicYearID = getDefaultAcademicYearID();
        $monthlyFees = collect();
        $months = range(1,  date('n'));
        $months = range(1, 12);
        foreach ($months as $month) {
            // Assigned fees
            $total = DB::table('class_fees')
                ->where('branch_id', $studentInfo->branch_id)
                ->where('academic_year_id', $academicYearID)
                ->where('student_class_id', $studentInfo->student_class_id)
                ->where('month', $month)
                ->sum('amount');
            // Paid fees
            $paid = DB::table('student_fees')
                ->where('branch_id', $studentInfo->branch_id)
                ->where('academic_year_id', $academicYearID)
                ->where('student_id', $studentInfo->id)
                ->where('month', $month)
                ->sum('total_amount');
            // Skip empty months
            if ($total == 0 && $paid == 0) {
                continue;
            }
            $monthlyFees->push([
                'month'        => $month,
                'total_amount' => number_format($total, 2, '.', ''),
                'paid_amount'  => number_format($paid, 2, '.', ''),
                'due_amount'   => number_format($total - $paid, 2, '.', ''),
            ]);
        }
        return $monthlyFees;
    }
    public function getDueFees()
    {
        $studentInfo = $this->getStudentInfo();
        $academicYearID = getDefaultAcademicYearID();
        $monthlyDueFees = collect();
        $months = range(1,  date('n'));
        $months = range(1, 12);
        foreach ($months as $month) {
            // Assigned fees
            $total = DB::table('class_fees')
                ->where('branch_id', $studentInfo->branch_id)
                ->where('academic_year_id', $academicYearID)
                ->where('student_class_id', $studentInfo->student_class_id)
                ->where('month', $month)
                ->sum('amount');
            // Paid fees
            $paid = DB::table('student_fees')
                ->where('branch_id', $studentInfo->branch_id)
                ->where('academic_year_id', $academicYearID)
                ->where('student_id', $studentInfo->id)
                ->where('month', $month)
                ->sum('total_amount');
            $due = $total - $paid;
            // ✅ Return only if due exists
            if ($due > 0) {
                $monthlyDueFees->push([
                    'month'        => $month,
                    'total_amount' => number_format($total, 2, '.', ''),
                    'paid_amount'  => number_format($paid, 2, '.', ''),
                    'due_amount'   => number_format($due, 2, '.', ''),
                ]);
            }
        }
        // dd($monthlyDueFees);
        return $monthlyDueFees;
    }
    public function getDueFees2()
    {
        $studentInfo = $this->getStudentInfo();
        $academicYearID = getDefaultAcademicYearID();
        $monthlyDueFees = collect();
        $months = range(1, 12);
        foreach ($months as $month) {
            // Assigned fees
            $total = DB::table('class_fees')
                ->where('branch_id', $studentInfo->branch_id)
                ->where('academic_year_id', $academicYearID)
                ->where('student_class_id', $studentInfo->student_class_id)
                ->where('month', $month)
                ->sum('amount');
            // Paid fees
            $paid = DB::table('student_fees')
                ->where('branch_id', $studentInfo->branch_id)
                ->where('academic_year_id', $academicYearID)
                ->where('student_id', $studentInfo->id)
                ->where('month', $month)
                ->sum('total_amount');
            // Skip empty months
            if ($total == 0 && $paid == 0) {
                continue;
            }
            $monthlyDueFees->push([
                'month'        => $month,
                'total_amount' => number_format($total, 2, '.', ''),
                'paid_amount'  => number_format($paid, 2, '.', ''),
                'due_amount'   => number_format($total - $paid, 2, '.', ''),
            ]);
        }
        return $monthlyDueFees;
        $studentInfo = $this->getStudentInfo();
        $academicYear = getDefaultAcademicYearValue();
        $fromDate = $academicYear . '-01-01';
        $toDate = $academicYear . '-12-31';
        // Class-wide fees query for due fees
        $classFeesQuery = DB::table('class_fees')
            ->join('fee_types as ft', 'class_fees.fee_type_id', '=', 'ft.id')
            ->join('student_classes as sc', 'class_fees.student_class_id', '=', 'sc.id')
            ->join('student_infos as si', 'sc.id', '=', 'si.student_class_id')
            ->leftJoin('invoice_items as ii', function ($join) use ($fromDate, $toDate) {
                $join->on('si.id', '=', 'ii.student_id')
                    ->whereColumn('class_fees.fee_type_id', 'ii.fee_type_id')
                    ->where('ii.payment_date', '>=', $fromDate)
                    ->where('ii.payment_date', '<=', $toDate);
            })
            ->whereNotIn('ft.code', ['Hostel', 'Transport'])
            ->where('si.student_id_no', $studentInfo->student_id_no)
            ->where('class_fees.branch_id', $studentInfo->branch_id)
            ->groupBy('ft.id', 'ft.title', 'ft.month', 'ft.code', 'class_fees.amount')
            ->havingRaw('class_fees.amount - COALESCE(SUM(ii.amount), 0) > 0')
            ->select([
                'ft.id as fee_type_id',
                'ft.title',
                'ft.month',
                'ft.code',
                'class_fees.amount as assigned_fee',
                DB::raw('COALESCE(SUM(ii.amount), 0) as paid_fee'),
                DB::raw('class_fees.amount - COALESCE(SUM(ii.amount), 0) as due_fee')
            ]);
        // Student-specific fees query for due fees
        $studentFeesQuery = DB::table('student_fees')
            ->join('fee_types as ft', 'student_fees.fee_type_id', '=', 'ft.id')
            ->leftJoin('invoice_items as ii', function ($join) use ($fromDate, $toDate) {
                $join->on('student_fees.student_id', '=', 'ii.student_id')
                    ->whereColumn('student_fees.fee_type_id', 'ii.fee_type_id')
                    ->where('ii.payment_date', '>=', $fromDate)
                    ->where('ii.payment_date', '<=', $toDate);
            })
            ->whereIn('ft.code', ['Hostel', 'Transport'])
            ->where('student_fees.student_id', $studentInfo->id)
            ->where('student_fees.status', 'active')
            ->groupBy('ft.id', 'ft.title', 'ft.month', 'ft.code', 'student_fees.amount')
            ->havingRaw('student_fees.amount - COALESCE(SUM(ii.amount), 0) > 0')
            ->select([
                'ft.id as fee_type_id',
                'ft.title',
                'ft.month',
                'ft.code',
                'student_fees.amount as assigned_fee',
                DB::raw('COALESCE(SUM(ii.amount), 0) as paid_fee'),
                DB::raw('student_fees.amount - COALESCE(SUM(ii.amount), 0) as due_fee')
            ]);
        // Combine queries using union
        return $classFeesQuery->unionAll($studentFeesQuery)->get();
    }
    public function getLeaveTypes()
    {
        $leaveTypes = LeaveType::all();
        return $leaveTypes;
    }
    public function storeApplyLeave(Request $request)
    {
        $leave = new Leave();
        $leave->user_id = auth()->user()->id;
        $leave->user_type = auth()->user()->user_type;
        $leave->leave_type_id = $request->leave_type;
        $leave->from_date = $request->from_date;
        $leave->to_date = $request->to_date;
        $leave->requested_days = $request->requested_days;
        $leave->leave_reason = $request->leave_reason;
        $leave->status = Status::PENDING;
        $leave->created_by = auth()->user()->id;
        $leave->save();
    }
    public function updateApplyLeave(Request $request, Leave $leave)
    {
        $leave->leave_type_id = $request->leave_type;
        $leave->from_date = $request->from_date;
        $leave->to_date = $request->to_date;
        $leave->requested_days = $request->requested_days;
        $leave->leave_reason = $request->leave_reason;
        $leave->save();
    }
    public function getLeaveHistory()
    {
        $studentInfo = $this->getStudentInfo();
        $leaveHistory = Leave::where('user_id', $studentInfo->user_id)->get();
        return $leaveHistory;
    }
    public function deleteApplyLeave(Leave $leave)
    {
        $leave->delete();
    }
}