<?php

namespace App\Services;

use App\Models\User;
use App\Jobs\SendSmsJob;
use App\Constants\Status;
use App\Constants\Settings;
use Illuminate\Support\Str;
use App\Models\GuardianInfo;
use Illuminate\Http\Request;
use App\Models\Hostel\Hostel;
use App\Models\Academic\Group;
use App\Models\Academic\Shift;
use App\Models\Academic\Branch;
use App\Models\Student\Sibling;
use App\Models\Academic\Section;
use App\Models\Hsc\GroupSubject;
use App\Models\Hostel\HostelRoom;
use App\Models\Result\ExamRecord;
use App\Filters\StudentInfoFilter;
use App\Models\Accounting\FeeType;
use App\Models\Transport\Stoppage;
use Illuminate\Support\Facades\DB;
use App\Models\Student\StudentInfo;
use App\Models\Hostel\StudentHostel;
use App\Models\HrPayroll\Occupation;
use Illuminate\Support\Facades\Hash;
use App\Models\Academic\AcademicYear;
use App\Models\Academic\StudentClass;
use App\Models\Accounting\StudentFee;
use App\Models\StudentInfoSubjectClass;
use App\Models\Scopes\BranchFilterScope;
use App\Models\Transport\TransportRoute;
use App\Models\Guardian\GuardianRelation;
use App\Models\Transport\StudentTransport;

class StudentInfoService
{


    private $globalPassword;
    protected $fileUploadService;

    public function __construct(FileUploadService $fileUploadService)
    {
        $this->fileUploadService = $fileUploadService;
    }

    protected function passwordGenerator()
    {
        return rand(10000000, 99999999); 
    }
    

    public function attendanceStatsCount($student_info_id)
    {

        $currentYear = date('Y'); // Get the current year
        $attendanceCount = DB::table('student_attendances')
            ->select('student_id',
                DB::raw('COUNT(CASE WHEN is_present = 1 THEN 1 END) as present_count'),
                DB::raw('COUNT(CASE WHEN is_absent = 1 THEN 1 END) as absent_count'),
                DB::raw('COUNT(CASE WHEN is_late = 1 THEN 1 END) as late_count'),
                DB::raw('COUNT(*) as total_days')
            )
            ->where('student_id', $student_info_id) // Filter by specific student
            ->whereYear('punch_date', $currentYear) // Filter by current year
            ->get();
        return $attendanceCount;
    }

    public function getStudentResults($student_info_id)
    {
        $examMarks = ExamRecord::where('student_id', $student_info_id)
            ->with('subject', 'exam', 'academicYear') // Eager load academicYear relationship
            ->orderBy('exam_id','desc')
            ->get()
            ->groupBy('year')
            ->map(function ($yearGroup) {
                return $yearGroup->groupBy('exam_id');
            });                       
                               // dd($examMarks);
        return $examMarks;
    }

    

    public function getStudentFees($student_id, $branch_id, $primary_id, $class_id)
    {
        $academicYear = AcademicYear::where('year', date('Y'))->first();

        // Check if year exists
        if (!$academicYear) {
            return []; // Or handle error appropriately
        }

        // $academicYearID = $academicYear->id;

        // $rows = DB::select("
        //         SELECT 
        //             t1.month,
        //             MONTHNAME(STR_TO_DATE(t1.month, '%m')) AS month_name,
        //             t2.code AS fee_type_name,
        //             t1.amount,
        //             CASE 
        //                 WHEN EXISTS (
        //                     SELECT 1
        //                     FROM invoice_items t3
        //                     WHERE t3.student_id = $primary_id
        //                     AND t3.fee_type_id = t1.fee_type_id
        //                     AND t3.academic_year_id = t1.academic_year_id
        //                     AND t3.month = t1.month
        //                 ) THEN 'Paid'
        //                 ELSE 'Due'
        //             END AS payment_status
        //         FROM class_fees t1
        //         LEFT JOIN fee_types t2 ON t1.fee_type_id = t2.id
        //         WHERE t1.student_class_id = $class_id
        //         AND t1.academic_year_id = $academicYearID
        //         AND t1.department_id = $department_id
        //         AND t1.branch_id = $branch_id
        //         ORDER BY t1.month, t2.code
        //     ");

        // // dd($rows);

        // $grouped = [];

        // foreach ($rows as $row) {
        //     $month = $row->month; // use month number as key to avoid duplication
        //     $month_name = $row->month_name;
        //     $payment_status = $row->payment_status;

        //     if (!isset($grouped[$month])) {
        //         $grouped[$month] = [
        //             'month' => $month_name,
        //             'total_amount' => 0,
        //             'payment_status' => $payment_status,
        //             'fee_types' => [],
        //         ];
        //     }

        //     $grouped[$month]['fee_types'][] = [
        //         'fee_type_name' => $row->fee_type_name,
        //         'amount' => (float) $row->amount,
        //     ];

        //     $grouped[$month]['total_amount'] += (float) $row->amount;
        // }

        // $final = array_values($grouped);
        
        return [];
    }



    public function getStudents(array $queryParams = [])
    {
        if(empty($queryParams)){
            return [];
        }
        $queryBuilder = StudentInfo::with(['branch', 'shift', 'section', 'studentClass', 'studentHostel', 'studentTransport'])
            ->orderBy('class_roll', 'asc')->latest();

        $students = resolve(StudentInfoFilter::class)->getResults([
            'builder' => $queryBuilder,
            'params' => $queryParams
        ]);
        
        return $students->get();
    }

    public function createStudentInfo(Request $request)
    {
        // $classRoll = $this->generateClassRoll($request->branch_id, $request->student_class_id, $request->shift_id, $request->section_id);
        // dd($request->all(), $classRoll );


        DB::beginTransaction();
        try {
            $this->globalPassword = $this->passwordGenerator();

            $fileName = $this->uploadProfileImage($request);
            $userData = $this->prepareUserData($request);
            $userID = User::insertGetId($userData);
            $requestData = $request->except(['elective_subjects', 'optional_subject']);
            $modifiedRequest = new \Illuminate\Http\Request($requestData);

            $studentInfoData = $this->prepareStudentInfoData($modifiedRequest, $fileName, $userID);
            
        $defaultYearID = getDefaultAcademicYearID2();

            // dd($studentInfoData, $this->generateStudentId(),$defaultYearID );
            //academic_year_id
            $studentInfoData['created_by'] = auth()->user()->id;
            $studentInfoData['academic_year_id'] = $defaultYearID;
            $studentInfoData['student_id_no'] = $this->generateStudentId();
            $studentInfoData['class_roll'] = $this->generateClassRoll($request->branch_id, $request->student_class_id, $request->shift_id, $request->section_id);
            $student = StudentInfo::create($studentInfoData);

            $student_create_sms = get_setting_value(Settings::STUDENT_CREATE_SMS) ? get_setting_value(Settings::STUDENT_CREATE_SMS) : '0';

            if($student_create_sms == '1'){
                $password = $this->globalPassword;
                $successCount = 0;
                $failCount = 0;
                $invalidNumbers = [];
                $number = $request->mobile_no;
                $school_short_name = get_setting_value(Settings::SCHOOL_SHORT_NAME) ? get_setting_value(Settings::SCHOOL_SHORT_NAME) : 'N/A';

                $studentClassName = $student->studentClass ? $student->studentClass->class_name : 'N/A';
                $studentSection = $student->section ? $student->section->section_name : 'N/A';
                $student_id = $student->student_id_no ?? 'N/A';

                $smsBody = "Dear ".$student->name.",\n";
                $smsBody .= "Your account has been created. You are now a student of Class ".$studentClassName.", Section ".$studentSection.".\n";
                $smsBody .= "Student ID: ".$student_id."\n";
                $smsBody .= "Password: ".$password."\n";
                $smsBody .= "Please keep this information safely.\n\n";
                $smsBody .= "Regards,\n";
                $smsBody .= $school_short_name;

                if (isValidBangladeshiNumber($number)) {
                    SendSmsJob::dispatch($number, $smsBody, 'student', ['branch_id' => auth()->user()->branch_id]);
                    $successCount++;
                } else {
                    $invalidNumbers[] = $number;
                    $failCount++;
                }
            }

            if($request->student_class_id == '13' || $request->student_class_id == '14' ){
                // Initialize
                $subjectMap = []; // key: subject_id, value: subject_type

                // Elective Subjects
                if (is_array($request->elective_subjects)) {
                    foreach (array_filter($request->elective_subjects) as $subjectId) {
                        $subjectMap[$subjectId] = 'elective';
                    }
                }

                // Optional Subject
                if (!empty($request->optional_subject)) {
                    $subjectMap[$request->optional_subject] = 'optional';
                }

                // Compulsory Subjects
                $compulsorySubjects = GroupSubject::where('type', GroupSubject::COMPULSORY)->get();
                foreach ($compulsorySubjects as $compulsory) {
                    $subjectMap[$compulsory->subject_id] = 'compulsory';
                }

                // Insert or update
                foreach ($subjectMap as $subjectId => $subjectType) {
                    StudentInfoSubjectClass::create([
                        'student_info_id'   => $student->id,
                        'subject_id'        => $subjectId,
                        'student_class_id'  => $request->student_class_id,
                        'subject_type'      => $subjectType,
                    ]);

                }


            }

            $this->storeHostelData($request, $student->id);
            $this->storeTransportData($request, $student->id);
            $guardianData = $this->prepareGuardianData($request, $student->id);
            GuardianInfo::create($guardianData);
            $this->storeSiblingData($request, $student->id);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function updateStudentInfo(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $student = StudentInfo::find($id);

            $requestData = $request->except(['elective_subjects', 'optional_subject']);
            $modifiedRequest = new \Illuminate\Http\Request($requestData);

            $fileName = $this->uploadProfileImage($request, $student->profile_image);
            $userData = $this->prepareUserData($request, $id);
            $student->user->update($userData);
            $studentInfoData = $this->prepareStudentInfoData($modifiedRequest, $fileName);
            $studentInfoData['updated_by'] = auth()->user()->id;
            $student->update($studentInfoData);

            if($request->student_class_id == '13' || $request->student_class_id == '14'){

                // Initialize
                $subjectMap = []; // key: subject_id, value: subject_type

                // Elective Subjects
                if (is_array($request->elective_subjects)) {
                    foreach (array_filter($request->elective_subjects) as $subjectId) {
                        $subjectMap[$subjectId] = 'elective';
                    }
                }

                // Optional Subject
                if (!empty($request->optional_subject)) {
                    $subjectMap[$request->optional_subject] = 'optional';
                }

                // Compulsory Subjects
                $compulsorySubjects = GroupSubject::where('type', GroupSubject::COMPULSORY)->get();
                foreach ($compulsorySubjects as $compulsory) {
                    $subjectMap[$compulsory->subject_id] = 'compulsory';
                }

                // Insert or update
                foreach ($subjectMap as $subjectId => $subjectType) {
                    StudentInfoSubjectClass::updateOrCreate(
                        [
                            'student_info_id'   => $student->id,
                            'subject_id'        => $subjectId,
                            'student_class_id'  => $request->student_class_id,
                        ],
                        [
                            'subject_type'      => $subjectType,
                        ]
                    );
                }

            }


            $this->storeHostelData($request, $id);
            $this->storeTransportData($request, $id);
            $guardianData = $this->prepareGuardianData($request, $id);
            $student->guardianInfo->update($guardianData);
            $this->updateSiblingData($request, $student->id);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function storeHostelData(Request $request, $student_id)
    {
        $hostel_id = $request->input('hostel');
        $hostel_floor_id = $request->input('hostel_floor_id');
        $hostel_room_id = $request->input('room_number');

        $hostel_room = HostelRoom::with('roomType')->where('id', $hostel_room_id)->first();
        $studentHostel = StudentHostel::where('student_id', $student_id)->first();
        $hostelFeeTypes = FeeType::where('code', 'Hostel')->get();
        if (empty($hostel_id) && empty($hostel_floor_id) && empty($hostel_room_id) && !empty($studentHostel)) {
            $studentHostel->delete();
        } elseif(!empty($hostel_id) && !empty($hostel_floor_id) && !empty($hostel_room_id)) {
            if ($studentHostel) {
                $studentHostel->update([
                    'hostel_id' => $hostel_id,
                    'hostel_floor_id' => $hostel_floor_id,
                    'hostel_room_id' => $hostel_room_id,
                    'room_type_id' => $hostel_room->roomType->room_type_id,
                    'cost' => $hostel_room->roomType->cost_per_bed,
                    'status' => Status::ASSIGNED,
                ]);
                foreach ($hostelFeeTypes as $hostelFeeType) {
                    StudentFee::where('student_id', $student_id)->where('fee_type_id', $hostelFeeType->id)->update([
                        'amount' => $hostel_room->roomType->cost_per_bed,
                        'month' => $hostelFeeType->month,
                    ]);
                }
            } else {
                StudentHostel::create([
                    'student_id' => $student_id,
                    'hostel_id' => $hostel_id,
                    'hostel_floor_id' => $hostel_floor_id,
                    'hostel_room_id' => $hostel_room_id,
                    'status' => Status::ASSIGNED,
                ]);

                foreach ($hostelFeeTypes as $hostelFeeType) {
                    StudentFee::create([
                        'fee_type_id' => $hostelFeeType->id,
                        'student_id' => $student_id,
                        'month' => $hostelFeeType->month,
                        'amount' => $hostel_room->roomType->cost_per_bed,
                    ]);
                }
            }
        }
    }

    public function storeTransportData(Request $request, $student_id)
    {
        $route_id = $request->input('route_id');
        $stoppage_id = $request->input('stoppage_id');

        $stoppage = Stoppage::where('id', $stoppage_id)->first();

        $studentTransport = StudentTransport::where('student_id', $student_id)->first();

        $transportFeeTypes = FeeType::where('code', 'Transport')->get();

        if (empty($route_id) && empty($stoppage_id) && !empty($studentTransport)) {
            $studentTransport->delete();
        } elseif(!empty($route_id) && !empty($stoppage_id)) {

            if (!$studentTransport) {
                StudentTransport::create([
                    'student_id' => $student_id,
                    'route_id' => $route_id,
                    'stoppage_id' => $stoppage_id,
                    'monthly_cost' => $stoppage->monthly_cost,
                ]);
                foreach ($transportFeeTypes as $transportFeeType) {
                    StudentFee::create([
                        'fee_type_id' => $transportFeeType->id,
                        'student_id' => $student_id,
                        'amount' => $stoppage->monthly_cost,
                    ]);
                }
            } elseif ($studentTransport) {
                StudentTransport::where('student_id', $student_id)->update([
                    'route_id' => $route_id,
                    'stoppage_id' => $stoppage_id,
                    'monthly_cost' => $stoppage->monthly_cost,
                ]);
                foreach ($transportFeeTypes as $transportFeeType) {
                    StudentFee::where('student_id', $student_id)->where('fee_type_id', $transportFeeType->id)->update([
                        'amount' => $stoppage->monthly_cost,
                    ]);
                }
            }
        }
    }

    public function getStudentNameWithID($studentIdNo, $students)
    {
        if ($studentIdNo && isset($students[0])) {
            return "{$students[0]->first_name} {$students[0]->last_name} ({$studentIdNo})";
        }
        return "";
    }

    private function uploadProfileImage(Request $request, $existingFile = '')
    {
        if ($request->hasFile('profile_image')) {
            $file = $request->file('profile_image');
            return $this->fileUploadService->upload($file, 'uploads/student', $existingFile);
        }
        return "";
    }

    private function prepareUserData(Request $request, $userID = '')
    {
        $pass = $this->globalPassword;
        $fullName = "$request->first_name $request->last_name";
        $fullNameStr = str_replace('.', '-', $fullName);
        $slug = Str::slug($fullNameStr);
        if ($userID=='') {
            $this->checkForExistingStudent($request, $slug);
        }
        return [
            'first_name' => $request->input('first_name'),
            'last_name' => $request->input('last_name'),
            'name' => $fullName,
            'slug' => $slug,
            'user_type' => 'student',
            'branch_id' => $request->branch_id,
            'phone_number' => $request->mobile_no,
            'email' => $request->email,
            'password' => Hash::make($pass)
        ];
    }

    private function checkForExistingStudent(Request $request, $slug)
    {
        $checkPreviousData = User::where('phone_number', $request->mobile_no)
            ->where('slug', $slug)
            ->where('user_type', 'student')
            ->first();

        if ($checkPreviousData) {
            throw new \Exception("Student name $request->first_name $request->last_name and mobile number $request->mobile_no is already created.");
        }
    }

    private function prepareStudentInfoData(Request $request, $fileName, $userID = '')
    {
        $studentInfoData = $request->except([
            'profile_image',
            'hostel',
            'hostel_floor_id',
            'room_number',
            'route_id',
            'stoppage_id',
            'sibling_id_no',
            // 'student_id_no',
        ]);

        $studentInfoData['name'] = "$request->first_name $request->last_name";

        if (!empty($fileName)) {
            $studentInfoData['profile_image'] = $fileName;
        }
        if (!empty($userID)) {
            $studentInfoData['user_id'] = $userID;
        }
        return $studentInfoData;
    }

    private function storeSiblingData(Request $request, $student_id)
    {
        if (isset($request->sibling_id_no)) {
            // Ensure sibling_ids is an array and not null
            $siblingIds = $request->sibling_id_no ?? [];
            $siblingData = [];
    
            foreach ($siblingIds as $sibling_id) {
                // Find the sibling by student_id_no
                $siblingInfo = StudentInfo::where('student_id_no', $sibling_id)->first();
    
                if ($siblingInfo) {
                    // Store both directions of the relationship (student -> sibling and sibling -> student)
                    $siblingData[] = [
                        'student_id' => $student_id,
                        'sibling_id' => $siblingInfo->id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                    $siblingData[] = [
                        'student_id' => $siblingInfo->id,
                        'sibling_id' => $student_id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }
            }
    
            // Insert both directions of the relationship at once
            Sibling::insert($siblingData);
        }
    }
    

    private function updateSiblingData(Request $request, $student_id)
    {
        if (isset($request->sibling_id_no)) {
            // Delete old siblings from the `siblings` table
            Sibling::where('student_id', $student_id)->orWhere('sibling_id', $student_id)->delete();
    
            // Ensure sibling_ids is an array and not null
            $siblingIds = $request->sibling_id_no ?? [];
            $siblingData = [];
    
            foreach ($siblingIds as $sibling_id) {
                // Find the sibling by student_id_no
                $siblingInfo = StudentInfo::where('student_id_no', $sibling_id)->first();
    
                if ($siblingInfo) {
                    // Store both directions of the relationship (student -> sibling and sibling -> student)
                    $siblingData[] = [
                        'student_id' => $student_id,
                        'sibling_id' => $siblingInfo->id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                    $siblingData[] = [
                        'student_id' => $siblingInfo->id,
                        'sibling_id' => $student_id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }
            }
    
            // Insert both directions of the relationship at once
            Sibling::insert($siblingData);
        }
    }
    

    private function prepareGuardianData(Request $request, $studentID = '')
    {
        return [
            'student_id' => $studentID,
            'guardian_name' => $request->input('guardian_name'),
            'guardian_occupation' => $request->input('guardian_occupation'),
            'guardian_mobile' => $request->input('guardian_mobile'),
            'guardian_email' => $request->input('guardian_email'),
            'guardian_relationship' => $request->input('guardian_relationship'),
            'guardian_address' => $request->input('guardian_address'),
            'password' => Hash::make('12345678')
        ];
    }

    public function searchStudent($query)
    {
        $students = StudentInfo::with(['branch', 'shift', 'section', 'studentClass'])
            ->where('student_id_no', 'LIKE', "%{$query}%")
            ->orWhere('mobile_no', 'LIKE', "%{$query}%")
            ->take(15)
            ->where('status', '!=', Status::GRADUATED)
            ->get(['id','student_id_no', 'mobile_no', 'first_name', 'last_name']);
        return $students;
    }

    public function getRelatedData()
    {
        return [
            'studentClasses' => StudentClass::pluck('class_name', 'id'),
            'groups' => Group::pluck('group_name', 'id'),
            'shifts' => Shift::pluck('shift_name', 'id'),
            'sections' => Section::pluck('section_name', 'id'),
        ];
    }

    public function getFormData()
    {
        return [
            'studentClasses' => StudentClass::pluck('class_name', 'id'),
            'groups' => Group::pluck('group_name', 'id'),
            'shifts' => Shift::pluck('shift_name', 'id'),
            'sections' => Section::pluck('section_name', 'id'),
            'branches' => Branch::pluck('branch_name', 'id'),
            'defaultBranch' => auth()->user()->branch_id,
            'occupations' => Occupation::pluck('name', 'id'),
            'guardian_relations' => GuardianRelation::pluck('name', 'id'),
            'hostels' => Hostel::pluck('name', 'id'),
            'transport_routes' => TransportRoute::pluck('route_name', 'id'),
            'student_id_no' => $this->generateStudentId(),
        ];
    }

    public function deleteStudentInfo($id)
    {
        DB::beginTransaction();
        try {
            $student = StudentInfo::find($id);
            if ($student) {
                $student->guardianInfo?->delete();
                $student->delete();
                $student->user->delete();
            }
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }


    public function generateStudentId($year_para = '')
    {
        $yearType = get_setting_value('student_id_year');
        $defaultYear = getDefaultAcademicYearValue2();
        // dd($defaultYear);
    
        $year = $year_para !='' ?$year_para : substr($defaultYear, -2); //$this->determineYear($yearType);
        // Temporarily remove the global scope to get the last student ID across all branches
        $lastStudentIdNo = StudentInfo::withoutGlobalScope(BranchFilterScope::class)
            ->where('student_id_no', 'LIKE', $year . '%')
            ->orderBy('student_id_no', 'desc')
            ->first();
        
        // If there are no existing student_id for this year, start with 1001
        if (!$lastStudentIdNo) {
            return $year . '1001';
        }
    
        // Extract the last student ID no for this year
        $lastNumericPart = substr($lastStudentIdNo->student_id_no, 2);
        $studentIdNo = $year . (str_pad((int)$lastNumericPart + 1, 4, '0', STR_PAD_LEFT));
    
        // Ensure uniqueness
        while (StudentInfo::withoutGlobalScope(BranchFilterScope::class)
            ->where('student_id_no', $studentIdNo)
            ->exists()
        ) {
            // Increment the numeric part
            $lastNumericPart++;
            $studentIdNo = $year . (str_pad($lastNumericPart, 4, '0', STR_PAD_LEFT));
        }
    
        return $studentIdNo;
    }

    protected function determineYear($yearType = 'current'): string
    {
        return match($yearType) {
            'current' => date('y'),
            'next' => date('y', strtotime('+1 year')),
            default => date('y')
        };
    }

    function generateClassRoll($branch_id, $class_id, $shift_id = null, $section_id = null)
    {
        $query = StudentInfo::where('branch_id', $branch_id)
            ->where('student_class_id', $class_id);

        // shift nullable logic
        if (is_null($shift_id)) {
            $query->whereNull('shift_id');
        } else {
            $query->where('shift_id', $shift_id);
        }

        // section nullable logic
        if (is_null($section_id)) {
            $query->whereNull('section_id');
        } else {
            $query->where('section_id', $section_id);
        }

        // get max roll in this group
        $maxRoll = $query->max('class_roll');

        return $maxRoll ? $maxRoll + 1 : 1;
    }
}
