<?php

namespace App\Traits;

use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Collection;

trait ReportExportTrait 
{
    /**
     * Generate Excel or PDF export
     * 
     * @param string $type (excel|pdf)
     * @param Collection|array $data
     * @param array $columns
     * @param array $options
     * @return mixed
     */
    protected function generateExport($type, $data, $columns, array $options = [])
    {
        $defaults = [
            'filename' => 'report',
            'title' => 'Report',
            'orientation' => count($columns['columns']) > 10 ? 'landscape' : 'portrait',
            'view' => $type === 'excel' ? 'exports.excel-template' : 'exports.pdf-template',
            'paper_size' => 'a4'
        ];

        $options = array_merge($defaults, $options);

        if ($type === 'excel') {
            return $this->exportToExcel($data, $columns, $options);
        }
        if ($type === 'print') {
            return $this->exportToPrint($data, $columns, $options);
        }
        
        return $this->exportToPdf($data, $columns, $options);
    }

    private function exportToExcel($data, $columns, $options)
    {
        return Excel::download(
            new \App\Exports\GenericExport(
                $data,
                $columns['columns'],
                $options['title'],
                $options['subtitle'] ?? ''
            ),
            $options['filename'] . '.xlsx'
        );
    }

    private function exportToPdf($data, $columns, $options)
    {
        $pdf = Pdf::loadView($options['view'], [
            'data' => $data,
            'columns' => $columns['columns'],
            'title' => $options['title'],
            'subtitle' => $options['subtitle'] ?? ''
        ])
        ->setPaper($options['paper_size'], $options['orientation']);

        return $pdf->download($options['filename'] . '.pdf');
    }

    private function exportToPrint($data, $columns, $options)
    {
        return view($options['view'], [
            'data' => $data,
            'columns' => $columns['columns'],
            'title' => $options['title'],
            'subtitle' => $options['subtitle'] ?? ''
        ]);
    }

    protected function generatePrintView($data, $params = [], $options = [])
    {
        $viewData = [
            'data' => $data,
            'columns' => $params['columns'] ?? [],
            'title' => $options['title'] ?? 'Report',
            'subtitle' => $options['subtitle'] ?? null,
            'extra_info' => $options['extra_info'] ?? null
        ];

        // dd($data, $viewData, $params['columns']);
        return view('exports.print-template', $viewData);
    }
} 
